(ns com.vadelabs.xtdb-core.interface
  (:require
   [clojure.java.io :as io]
   [xtdb.api :as xt]))

(defn save-tx-fns! [node tx-fns]
  (let [db (xt/db node)]
    (when-some [tx (not-empty
                     (vec
                       (for [[k f] tx-fns
                             :let [new-doc {:xt/id k
                                            :xt/fn f}
                                   old-doc (xt/entity db k)]
                             :when (not= new-doc old-doc)]
                         [::xt/put new-doc])))]
      (xt/submit-tx node tx))))

(defn start-node
  [{:keys [topology dir opts jdbc-spec pool-opts kv-store tx-fns]
    :or {kv-store :rocksdb}}]
  (let [kv-store-fn (fn [basename]
                      {:kv-store {:xtdb/module (if (= kv-store :lmdb)
                                                 'xtdb.lmdb/->kv-store
                                                 'xtdb.rocksdb/->kv-store)
                                  :db-dir (io/file dir (str basename (when (= kv-store :lmdb)
                                                                       "-lmdb")))}})
        node (xt/start-node
               (merge (case topology
                        :standalone
                        {:xtdb/index-store    (kv-store-fn "index")
                         :xtdb/document-store (kv-store-fn "docs")
                         :xtdb/tx-log         (kv-store-fn "tx-log")}

                        :jdbc
                        {:xtdb/index-store (kv-store-fn "index")
                         :xtdb.jdbc/connection-pool {:dialect {:xtdb/module
                                                               'xtdb.jdbc.psql/->dialect}
                                                     :pool-opts pool-opts
                                                     :db-spec jdbc-spec}
                         :xtdb/tx-log {:xtdb/module 'xtdb.jdbc/->tx-log
                                       :connection-pool :xtdb.jdbc/connection-pool}
                         :xtdb/document-store {:xtdb/module 'xtdb.jdbc/->document-store
                                               :connection-pool :xtdb.jdbc/connection-pool}})
                 opts))
        f (future (xt/sync node))]
    (while (not (realized? f))
      (Thread/sleep 2000)
      (when-some [indexed (xt/latest-completed-tx node)]
        (println "Indexed" (pr-str indexed))))
    (when (not-empty tx-fns)
      (save-tx-fns! node tx-fns))
    node))

(defn q [db query & args]
  (when-not (= (count (:in query))
              (count args))
    (throw (ex-info (str "Incorrect number of query arguments. Expected "
                      (count (:in query))
                      " but got "
                      (count args)
                      ".")
             {})))
  (let [return-tuples (vector? (:find query))
        query (cond-> query
                (not return-tuples) (update :find vector))
        results (apply xt/q db query args)]
    (cond->> results
      (not return-tuples) (map first))))

(defn lazy-q
  [db query & args]
  (when-not (= (count (:in query))
              (dec (count args)))
    (throw (ex-info (str "Incorrect number of query arguments. Expected "
                      (count (:in query))
                      " but got "
                      (count args)
                      ".")
             {})))
  (let [f (last args)
        query-args (butlast args)
        return-tuples (vector? (:find query))
        query (cond-> query
                (not return-tuples) (update :find vector))]
    (with-open [results (apply xt/open-q db query query-args)]
      (f (cond->> (iterator-seq results)
           (not return-tuples) (map first))))))
