(ns com.vadelabs.adapter-postgres.dsl
  (:require
   [clojure.set :as cset]
   [com.vadelabs.utils-core.interface :as uc]))

(defn create-schema
  [schemas _ next-model]
  (let [schemas (if (vector? schemas) schemas [schemas])]
    (mapv (fn [schema]
            [:create-schema {:if-not-exists true}
             (get-in next-model [schema :name])])
      schemas)))

(defn create-enum
  [enums _ next-model]
  (let [enums (if (vector? enums) enums [enums])]
    (mapv
      (fn [enum]
        (into
          [:create-type {:as :enum} (get-in next-model [enum :name])]
          (get-in next-model [enum :options])))
      enums)))

(defn drop-schema
  [schemas prev-model _]
  (let [schemas (if (vector? schemas) schemas [schemas])]
    (mapv (fn [schema]
            [:drop-schema {:if-exists true}
             (get-in prev-model [schema :name])])
      schemas)))

(defn alter-schema
  [schemas prev-model next-model]
  (let [schemas (if (vector? schemas) schemas [schemas])]
    (mapv (fn [schema]
            [:alter-schema {:type :rename-to}
             (get-in prev-model [schema :name])
             (get-in next-model [schema :name])])
      schemas)))

(defn ^:private primary-key
  [{:keys [name fields]}]
  (when (seq fields)
    (into [:primary-key {} name]
      fields)))

(defn ^:private columns
  [cols]
  (map (fn [{:keys [name props type]}]
         [name props type])
    (vals cols)))

(defn create-table
  [tables _ next-model]
  (let [tables (if (vector? tables) tables [tables])]
    (mapv (fn [table]
            (let [columns (columns (get-in next-model [table :columns]))
                  primary-key (primary-key (get-in next-model [table :primary-key]))]
              (cond-> [:create-table {:if-not-exists true}
                       (get-in next-model [table :name])]
                (seq columns) (into columns)
                (seq primary-key) (conj primary-key))))
      tables)))

(defn ^:private add-column
  [table columns _ next-model]
  (let [columns (if (vector? columns) columns [columns])]
    (into [:add-column {:if-not-exists true} table]
      (mapv
        (fn [column]
          [(->> [column :name] (get-in next-model) uc/keywordize)
           (get-in next-model [column :props] {})
           (->> [column :type] (get-in next-model) uc/keywordize)])
        columns))))

(defn ^:private drop-column
  [table columns prev-model _]
  (let [columns (if (vector? columns) columns [columns])]
    (into [:drop-column {:if-exists true} table]
      (mapv
        (fn [column]
          [(->> [column :name] (get-in prev-model) uc/keywordize)])
        columns))))

(defn ^:private alter-column
  [table columns prev-model next-model]
  (let [columns (if (vector? columns) columns [columns])]
    (into [:alter-column {} table]
      [])))

(defn changes-columns
  [table prev-model next-model]
  (let [[alterations removals] (uc/diff prev-model next-model)
        changes (-> (set (keys alterations))
                  (cset/union (set (keys removals))))
        created (uc/created changes prev-model next-model)
        dropped (uc/dropped changes prev-model next-model)
        altered (uc/altered changes prev-model next-model)]
    (cond-> []
      (seq created) (into (add-column table created prev-model next-model))
      (seq dropped) (into (drop-column table dropped prev-model next-model))
      (seq altered) (into (alter-column table altered prev-model next-model)))))

(defn alter-table
  [tables prev-model next-model]
  (let [tables (if (vector? tables) tables [tables])]
    (mapv (fn [table]
            (let [columns (changes-columns
                            (get-in next-model [table :name])
                            (get-in prev-model [table :columns])
                            (get-in next-model [table :columns]))]
              (cond-> []
                (seq columns) (into columns))))
      tables)))
