(ns com.vadelabs.adapter-xtdb.main
  (:require
   [clojure.walk :as cwalk]
   [com.vadelabs.utils-core.interface :as uc]
   [com.vadelabs.utils-str.interface :as ustr]
   [com.wsscode.pathom3.connect.operation :as pco]
   [edn-query-language.core :as eql]
   [xtdb.api :as xt]))

(def transaction-functions
  {::delta-update '(fn [ctx [id {:keys [before after]}]]
                     (let [db (xtdb.api/db ctx)
                           entity (xtdb.api/entity db id)
                           before (when entity
                                    (reduce (fn [e [k v]]
                                              (cond-> e
                                                (and (some? v) (get e k)) (assoc k v)))
                                      entity
                                      before))]
                       [[:xtdb.api/match id before]
                        [:xtdb.api/put (merge entity after)]]))})

(defn ref?
  [attributes-map k]
  (= :attribute.type/ref (some-> k attributes-map :attribute/local-type)))

(defn ->delta-update-txs
  [doc-delta]
  (reduce
    (fn [acc [id delta]]
      (conj acc [::xt/fn ::delta-update [id delta]]))
    []
    doc-delta))

(defn is-ident?
  [value]
  (and (vector? value)
    (= 2 (count value))
    (keyword? (first value))
    (or (uuid? (second value))
      (uc/tempid? (second value)))))

(defn ident->id
  [[_attr key] tempid->realid]
  (if (uc/tempid? key)
    (tempid->realid key)
    key))

(defn idents->ids
  [doc tempid->realid env]
  (->> doc
    (reduce (fn [acc [k v]]
              (assoc acc k (cond
                             (is-ident? v) (ident->id v tempid->realid)

                             (and (vector? v) (every? is-ident? v)) (mapv #(ident->id % tempid->realid) v)

                             :else v)))
      {})))

(defn ->doc-delta
  "Converts attribute delta into document delta"
  [attr-delta tempid->realid env]
  (->> attr-delta
    (reduce (fn [d [[attr key] doc]]
              (let [key (get tempid->realid key key)]
                (assoc-in d [key] (->
                                    (reduce (fn [acc [attr {:keys [before after] :as value}]]
                                              (let [after (or after value)]
                                                (-> acc
                                                  (assoc-in [:before attr] before)
                                                  (assoc-in [:after attr] after))))
                                      {}
                                      doc)
                                    (assoc-in [:after attr] key)
                                    (assoc-in [:after :xt/id] key)
                                    (update :before idents->ids tempid->realid env)
                                    (update :after idents->ids tempid->realid env)))))
      {})))

(defn generate-next-id
  "Generate an id. You may pass a `suggested-id` as a UUID or a tempid. If it is a tempid and the ID column is a UUID, then
  the UUID *from* the tempid will be used. If the ID column is not a UUID then the suggested id is ignored."
  ([env k]
   (generate-next-id env k (uc/uuid-next)))
  ([env k suggested-id]
   (cond
     (uc/tempid? suggested-id) (:id suggested-id)
     (uuid? suggested-id) suggested-id
     :else (uc/uuid-next))))

(defn tempids->realids [env delta]
  (->> (keys delta)
    (keep (fn [[k maybe-tempid]]
            (when (uc/tempid? maybe-tempid)
              [maybe-tempid (generate-next-id env k maybe-tempid)])))
    (into {})))

(defn delta->txn
  [env delta]
  (let [tempid->realid (tempids->realids env delta)]
    {:tempid->realid tempid->realid
     :txn (-> delta
            (->doc-delta tempid->realid env)
            ->delta-update-txs)}))

(defn prepare-input
  [data]
  (cwalk/postwalk
    (fn [node]
      (if (uc/is-ident? node)
        (let [[kw val] node
              val (if (uuid? val) val (-> (ustr/str kw val) uc/uuid uc/tempid))]
          [kw val])
        node))
    data))

(defn save!
  [{:keys [node] :as aenv} action penv {:keys [data]}]
  (let [delta (-> data uc/normalize prepare-input)
        {:keys [tempid->realid txn]} (delta->txn aenv delta)
        result (atom {:tempids {}})
        tx (xt/submit-tx node txn)]
    (swap! result update :tempids merge tempid->realid)
    (xt/await-tx node tx)
    (when-not (xt/tx-committed? node tx)
      (println "Transaction failed!"))
    @result))

;; {studio/save! {:tempids {#fulcro/tempid["0f825c54-0928-53f7-9807-f2a49ecc9296"] #uuid "0f825c54-0928-53f7-9807-f2a49ecc9296", #fulcro/tempid["1896c143-aa74-5c44-b1d8-f9f13db9c93f"] #uuid "1896c143-aa74-5c44-b1d8-f9f13db9c93f", #fulcro/tempid["15e15699-f9f0-5271-83e8-f487bd4e8f1d"] #uuid "15e15699-f9f0-5271-83e8-f487bd4e8f1d", #fulcro/tempid["e4aad4a3-be71-57b5-93eb-14afd8507714"] #uuid "e4aad4a3-be71-57b5-93eb-14afd8507714", #fulcro/tempid["5a6f811e-1f0e-53d8-87f6-14c77dc250ed"] #uuid "5a6f811e-1f0e-53d8-87f6-14c77dc250ed", #fulcro/tempid["41aabf08-9aac-5d73-a7dd-43a4a6db85d2"] #uuid "41aabf08-9aac-5d73-a7dd-43a4a6db85d2", #fulcro/tempid["bead2ebf-3299-53aa-a07d-87bc386608ab"] #uuid "bead2ebf-3299-53aa-a07d-87bc386608ab", #fulcro/tempid["0a4e7bee-97ae-5d14-bab2-8db49bc8bd62"] #uuid "0a4e7bee-97ae-5d14-bab2-8db49bc8bd62", #fulcro/tempid["0028b855-e571-58c3-963b-7985f0a32426"] #uuid "0028b855-e571-58c3-963b-7985f0a32426"}}}

(defn save-entity!
  [{:keys [attributes-map node] :as aenv} action penv pparams]
  (save! aenv action penv pparams))

(defn delete-entity!
  "Delete the given entity, if possible."
  [{:keys [node]} action penv pparams]
  (let [pk (ffirst pparams)
        eid (get pparams pk)
        tx (xt/submit-tx node [[::xt/delete eid]])]
    (xt/await-tx node tx)))

(defn resolve-aliased-attrs
  [{:keys [attributes-map]} output]
  (cwalk/prewalk
    (fn [node]
      (if (qualified-keyword? node)
        (get-in attributes-map [node :attribute/remote-key])
        node))
    output))

(defn ^:private local-result
  [{:keys [attributes-map]} ast-nodes remote-result]
  (if (seq ast-nodes)
    (reduce
      (fn [acc {:keys [dispatch-key type]}]
        (let [{:attribute/keys [remote-key target]} (get attributes-map dispatch-key)
              result-value (or (get remote-result remote-key)
                             (get remote-result dispatch-key)) ;; Fallback to dispatch-key if result is nil from remote-key
              #_local-type #_(get-in attributes-map [target :attribute/local-type] local-type)
              #_result-value #_(uc/coerce local-type result-value)]
          (cond-> acc
            (= type :join) (assoc-in [dispatch-key target] result-value)
            (= type :prop) (assoc-in [dispatch-key] result-value))))
      {}
      ast-nodes)
    remote-result))

(defn ^:private local-results
  [adapter ast-nodes remote-results]
  (if (map? remote-results)
    (local-result adapter ast-nodes remote-results)
    (mapv (partial local-result adapter ast-nodes) remote-results)))

(defn ^:private where-clause
  [{:keys [attributes-map]} pinput]
  (if (seq pinput)
    (reduce-kv
      (fn [acc key value]
        (let [entity-key (get-in attributes-map [key :attribute/remote-key])]
          (conj acc ['?entity entity-key (or value '_)])))
      []
      pinput)
    [['?entity :xt/id '_]]))

(defn remote-query
  [aenv penv pinput desired-output]
  (let [desired-output (if (seq desired-output) desired-output ['*])
        {:keys [limit offset]} (pco/params penv)]
    (cond-> {:find [`(~'pull ~'?entity ~desired-output)]
             :where (where-clause aenv pinput)}
      limit (assoc :limit limit)
      offset (assoc :offset offset))))

(defn exec-query
  [{:keys [node]} query]
  (let [results (-> node
                  xt/db
                  (xt/q query))]
    (->> results
      (map first)
      (filterv seq))))

(defn user-shape
  [penv]
  (get-in penv
    [:com.wsscode.pathom3.connect.planner/graph
     :com.wsscode.pathom3.connect.planner/user-request-shape]))

(defn eql-query
  [user-shape {:action/keys [identifier] :as action}]
  (reduce-kv
    (fn [acc k v]
      (cond
        (= identifier k) acc
        (= :com.wsscode.pathom3.connect.runner/attribute-errors k) acc
        (seq v) (conj acc {k (eql-query v action)})
        :else (conj acc k)))
    []
    user-shape))

(defn ^:private ->remote-key
  [attributes-map qualified-key]
  (or (get-in attributes-map [qualified-key :attribute/remote-key])
    (uc/raise :type :not-found
      :code :remote-key-not-found
      :qualified-key qualified-key
      :hint (ustr/format "couldn't find remote key for %s" qualified-key))))

(defn local-query
  [{:keys [attributes-map]} {:action/keys [identifier]} tree]
  (cwalk/prewalk
    (fn [node]
      (if (and (qualified-keyword? node) (not= node identifier))
        (->remote-key attributes-map node)
        node))
    tree))

(defn local-result
  [{:keys [attributes-map] :as aenv} ast-nodes result-map]
  (reduce
    (fn [acc {:keys [children dispatch-key type]}]
      (let [{:attribute/keys [remote-key target]} (get attributes-map dispatch-key)
            result-value (or (get result-map remote-key)
                           (get result-map dispatch-key))]
        (cond-> acc
          (= type :join) (assoc dispatch-key (if (map? result-value)
                                               (local-result aenv children result-value)
                                               (mapv (partial local-result aenv children) result-value)))
          (= type :prop) (assoc dispatch-key result-value))))
    {}
    ast-nodes))

(defn local-results
  [aenv eql-query results]
  (let [{:keys [children]} (eql/query->ast eql-query)]
    (if (map? results)
      (local-result aenv children results)
      (mapv (partial local-result aenv children) results))))

(defn query
  [aenv {:action/keys [type identifier output] :as action} penv pinput]
  (let [eql-query (-> penv user-shape (eql-query action))
        eql-query (if (seq eql-query) eql-query output)
        local-query (local-query aenv action eql-query)
        remote-query (remote-query aenv penv pinput local-query)
        remote-results (exec-query aenv remote-query)
        local-results (local-results aenv eql-query remote-results)]
    #_(->> eql-query
        (local-query aenv action)
        (remote-query aenv penv pinput)
        (exec-query aenv)
        (local-results aenv eql-query))
    #_(tap> {:rr remote-results
             :lr local-results
             :nq nspaced-query
             :uq uneql-query
             :rq remote-query})
    (case type
      :list {identifier local-results}
      :ident (or (first local-results) {})
      (uc/raise :type :not-found
        :code :action-type-not-found
        :action action
        :hint (ustr/format "couldn't find action type %s" type)))))
