(ns com.vadelabs.adapter-rest.pathom
  (:require
   [com.vadelabs.datasource-core.interface :as dc]
   [com.vadelabs.utils-core.interface :as uc]
   [com.wsscode.pathom3.connect.operation :as pco]
   [clojure.data.json :as json]
   [clj-http.client :as http-client]
   [camel-snake-kebab.core :as csk]))

(defn ^:private mutation
  [aenv {:action/keys [params output identifier]} mutation-fn]
  (pco/mutation
    (cond-> {::pco/op-name (uc/symbolize identifier)
             ::pco/mutate mutation-fn}
      (seq params) (assoc ::pco/params params)
      (seq output) (assoc ::pco/output output))))

(defn ^:private optional-input
  [input]
  (mapv #(pco/? %) input))

(defn ^:private root-resolver
  [aenv {:action/keys [identifier input params output priority optionals]} resolver-fn]
  (let [input (->> optionals optional-input (into input))]
    (pco/resolver
      (cond-> {::pco/op-name (uc/symbolize identifier)
               ::pco/resolve resolver-fn}
        priority (assoc ::pco/priority priority)
        (seq output) (assoc ::pco/output [{(uc/keywordize identifier) output}])
        (seq input) (assoc ::pco/input input)
        (seq params) (assoc ::pco/params params)))))

(defn ^:private resolver
  [aenv {:action/keys [identifier input params output priority optionals]} resolver-fn]
  (let [input (->> optionals optional-input (into input))]
    (pco/resolver
      (cond-> {::pco/op-name (uc/symbolize identifier)
               ::pco/resolve resolver-fn}
        priority (assoc ::pco/priority priority)
        (seq output) (assoc ::pco/output output)
        (seq input) (assoc ::pco/input input)
        (seq params) (assoc ::pco/params params)))))

(defn response-for
  [{:keys [http-instance] :as aenv} action penv pparams]
  (let [provider (-> :identifier aenv namespace keyword)
        route-name (-> :action/identifier action name keyword)
        access-tokens (get-in penv [:ring/request :session :oauth2/access-tokens provider])
        http-instance (if access-tokens
                        (dc/oauth2-instance aenv {:route-name route-name :access-tokens access-tokens})
                        http-instance)
        _ (tap> {:request (-> http-instance
                            (dc/request-for route-name pparams))})
        resp (-> http-instance
               (dc/response-for route-name pparams)
               :body)]
    {(:action/identifier action) resp}))

(defmulti ^:private operation
  (fn [_aenv {:action/keys [type]}]
    type))

(defmethod operation :post
  [aenv action]
  (mutation aenv action (partial response-for aenv action)))

(defmethod operation :get
  [aenv action]
  (root-resolver aenv action (partial response-for aenv action)))

(defn operations
  [{:keys [actions] :as aenv}]
  (mapv (partial operation aenv)

    [{:action/identifier :userinfo
      :action/type :get
      :action/output [:Id :Status :ProviderName :DateTimeUTC :Organisations]}
     {:action/identifier :tenants
      :action/type :get
      :action/output [:UserID :FirstName]}
     {:action/identifier :contacts
      :action/type :get
      :action/output [:Id :Status :Contacts]}
     {:action/identifier :accounts
      :action/type :get
      :action/output [:Id :Status :Accounts]}
     {:action/identifier :items
      :action/type :get
      :action/output [:Id :Status :Items]}
     {:action/identifier :invoices
      :action/type :get
      :action/output [:Id :Status :Invoices]}
     {:action/identifier :create-invoice
      :action/type :post
      :action/params [:Type :Contact :LineItems :Date :DueDate :Reference :Status]
      :action/output [:Id :Status :ProviderName :Invoices]}]))
