(ns com.vadelabs.sql-core.nodes.column
  (:require
   [com.vadelabs.sql-core.ast :as ast]
   [com.vadelabs.utils-core.interface :as uc]))

(defn ^:private sql-enum-type
  [schema table column-name column-type]
  (->> [table column-name column-type]
    (map uc/namify)
    (uc/str-join "_")
    (uc/prefix-str "." schema)))

(defn ^:private sql-type
  [{:keys [cardinality table schema]} column-name column-type]
  (let [data-type (if (vector? column-type) (first column-type) column-type)
        precision (when (vector? column-type) (second column-type))]
    (uc/sql-join ""
      (cond-> []
        (= :enum data-type) (conj (sql-enum-type schema table column-name data-type))
        (not= :enum data-type) (conj (uc/sql-literal data-type))
        (= :many cardinality) (conj "[]")
        precision (conj (uc/str-format "(%s)" precision))))))

(defn format-default
  [env default]
  (cond
    (= default :now) "now()"
    (= default :gen-random-uuid) "gen_random_uuid()"
    :else default))

(defmethod ast/compile [:postgres :column]
  [env [_ {:keys [column-name props column-type]}]]
  (let [{:keys [null not-null unsigned primary-key
                unique default foreign-key]} props]
    (uc/sql-join " "
      (cond-> [(uc/sql-name env column-name) (sql-type props column-name column-type)]
        null (into [(uc/sql-literal :null)])
        not-null (into [(uc/sql-literal :not-null)])
        default (into ["DEFAULT" (format-default env default)])
        primary-key (into [(uc/sql-literal :primary-key)])
        unique (into [(uc/sql-literal :unique)])
        unsigned (into [(uc/sql-literal :unsigned)])))))

(comment

  (ast/format
    [:col-one {:default :now :not-null true} :uuid])
  ;; => ["col_one UUID NOT NULL DEFAULT now()"]

  :rcf)
