(ns com.vadelabs.adapter-core.interface
  (:require
   [com.vadelabs.utils-core.interface :as uc]
   [com.wsscode.pathom.viz.ws-connector.core :as pvc]
   [com.wsscode.pathom.viz.ws-connector.pathom3 :as p.connector]
   [com.wsscode.pathom3.connect.built-in.plugins :as pbip]
   [com.wsscode.pathom3.plugin :as p.plugin]))

(defn ^:private dispatch-func
  ([_genv {:adapter/keys [type]}]
   type))

(defmulti prepare-env
  #'dispatch-func)

(defmulti initiate-env!
  #'dispatch-func)

(defmulti halt-env!
  #'dispatch-func)

(defmulti sync-schema!
  #'dispatch-func)

(defmulti pathom-env
  #'dispatch-func)

(defmulti enrich-attribute
  (fn [{:adapter/keys [type]} _attribute]
    type))

(defmulti default-actions
  (fn [{:adapter/keys [type]} _attributes]
    type))

(defn initiate!
  "Initiates all adapters and combines them into a global pathom environment"
  [{:keys [parser] :as gopts} adapters]
  (let [adapters (if (map? adapters) [adapters] adapters)
        genv  (reduce (fn [aenv adapter]
                        (-> aenv
                          (prepare-env adapter)
                          (initiate-env! adapter)
                          (sync-schema! adapter)
                          (pathom-env adapter)))
                gopts
                adapters)
        genv (cond-> genv
               (p.plugin/register pbip/mutation-resolve-params)
               (assoc :com.wsscode.pathom3.error/lenient-mode? true ::adapters adapters)
               parser (p.connector/connect-env {::pvc/parser-id parser}))]
    genv))

(defn halt!
  "Halts the datasource adapter if possible"
  [{::keys [adapters] :as genv}]
  (for [adapter adapters]
    (halt-env! genv adapter)))

(defn ^:private attributes-map
  [attrs]
  (->> attrs
    (mapcat (fn [{:attribute/keys [local-key qualified-key] :as attr}]
              [[local-key attr]
               (when qualified-key [qualified-key attr])]))
    (into {})))

(defmethod sync-schema! :default
  [genv _]
  genv)

(defmethod halt-env! :default
  [genv _]
  genv)

(defmethod default-actions :default
  [_ _]
  [])

(defmethod prepare-env :default
  [genv {:adapter/keys [actions attributes qualify-attributes nspace identifier type id config]
         :or {actions []}
         :as adapter}]
  (let [adapter (assoc adapter :adapter/attributes-map (attributes-map attributes))
        attributes (mapv (partial enrich-attribute adapter) attributes)
        actions (into actions (default-actions adapter attributes))]
    (cond-> genv
      (seq actions) (assoc-in [type id :actions] actions)
      (seq attributes) (assoc-in [type id :attributes] attributes)
      (seq attributes) (assoc-in [type id :attributes-map] (attributes-map attributes))
      nspace (assoc-in [type id :migration-history] (uc/prefix-keyword "." nspace :migration-history))
      nspace (assoc-in [type id :nspace] nspace)
      type (assoc-in [type id :dialect] (-> type name keyword))
      identifier (assoc-in [type id :identifier] identifier)
      (seq config) (assoc-in [type id :config] (or config {}))
      (boolean? qualify-attributes) (assoc-in [type id :qualify-attributes] qualify-attributes))))

(defmethod enrich-attribute :default
  [{:adapter/keys [qualify-attributes nspace]}
   {:attribute/keys [id local-key local-type cardinality options default doc target group
                     unique primary-key remote-ns remote-entity remote-key remote-type]}]
  (let [id (or id (uc/uuid local-key))
        qualified-key (if qualify-attributes (uc/keywordize nspace local-key) local-key)
        qualified-target (when target (if qualify-attributes (uc/keywordize nspace target) target))
        cardinality (or cardinality :one)
        group (or group :other)]
    (cond->
      {:attribute/id id
       :attribute/local-key local-key
       :attribute/local-type local-type
       :attribute/group group
       :attribute/qualified-key qualified-key
       :attribute/cardinality cardinality
       :attribute/doc (or doc "")}
      (boolean? unique) (assoc :attribute/unique unique)
      remote-ns (assoc :attribute/remote-ns remote-ns)
      remote-entity (assoc :attribute/remote-entity remote-entity)
      remote-key (assoc :attribute/remote-key remote-key)
      remote-type (assoc :attribute/remote-type remote-type)
      (seq options) (assoc :attribute/options options)
      default (assoc :attribute/default default)
      primary-key (assoc :attribute/primary-key primary-key)
      target (assoc :attribute/target target)
      qualified-target (assoc :attribute/qualified-target qualified-target))))
