(ns com.vadelabs.datasource-core.oauth2
  (:require
   [clj-http.client :as http-client]
   [clj-time.core :as time]
   [com.vadelabs.utils-core.interface :as uc]
   [com.vadelabs.utils-str.interface :as ustr]
   [crypto.random :as random]
   [ring.middleware.params :refer [wrap-params]]
   [ring.util.codec :as codec]
   [ring.util.response :as resp])
  (:import [java.net URI]))

(defn ^:private scopes
  [{:keys [scopes]}]
  (ustr/join " " (map uc/namify scopes)))

(defn ^:private authorize-uri
  [{:keys [authorize-uri client-id redirect-uri] :as provider} state]
  (ustr/str authorize-uri
    (if (ustr/includes? authorize-uri "?") "&" "?")
    (codec/form-encode {:response_type "code"
                        :client_id client-id
                        :redirect_uri redirect-uri
                        :scope (scopes provider)
                        :state state})))

(defn ^:private random-state []
  (-> (random/base64 9)
    (ustr/replace "+" "-")
    (ustr/replace "/" "_")))

(defn ^:private make-launch-handler
  [provider]
  (fn [request]
    (let [state (random-state)
          new-session (assoc (:session request) ::authorize-state state)]
      (-> provider
        (authorize-uri state)
        resp/redirect
        (assoc :session new-session)))))

(defn ^:private state-matches?
  [request]
  (= (get-in request [:session ::authorize-state])
    (get-in request [:query-params "state"])))

(defn ^:private coerce-to-int
  [n]
  (if (string? n)
    (Integer/parseInt n)
    n))

(defn ^:privade format-access-token
  [{:keys [body]}]
  (let [{:keys [access_token expires_in refresh_token id_token]} body]
    (cond-> {:token access_token
             :extra-data (dissoc body :access_token :expires_in :refresh_token :id_token)}
      expires_in (assoc :expires (-> expires_in
                                   coerce-to-int
                                   time/seconds
                                   time/from-now))
      refresh_token (assoc :refresh-token refresh_token)
      id_token (assoc :id-token id_token))))

(defn ^:private get-authorization-code
  [request]
  (get-in request [:query-params "code"]))

(defn ^:private request-params
  [provider request]
  {:grant_type "authorization_code"
   :code (get-authorization-code request)
   :redirect_uri (:redirect-uri provider)})

(defn ^:private add-header-credentials
  [options client-id client-secret]
  (assoc options :basic-auth [client-id client-secret]))

(defn ^:private add-form-credentials
  [options client-id client-secret]
  (assoc options :form-params (-> (:form-params options)
                                (merge {:client_id client-id
                                        :client_secret client-secret}))))

(defn ^:private get-access-token
  [{:keys [access-token-uri client-id client-secret basic-auth?] :as provider} request]
  (format-access-token
    (http-client/post access-token-uri
      (cond-> {:accept :json :as :json
               :form-params (request-params provider request)}
        basic-auth? (add-header-credentials client-id client-secret)
        (not basic-auth?) (add-form-credentials client-id client-secret)))))

(defn ^:private state-mismatch-handler
  [_]
  {:status 400 :headers {} :body "State mismatch"})

(defn ^:privade no-auth-code-handler
  [_]
  {:status 400 :headers {} :body "No authorization code"})

(defn ^:private make-redirect-handler
  [{:keys [id landing-uri] :as provider}]
  (let [state-mismatch-handler (:state-mismatch-handler provider state-mismatch-handler)
        no-auth-code-handler (:no-auth-code-handler provider no-auth-code-handler)]
    (fn [{:keys [session] :as request}]
      (cond
        (not (state-matches? request))
        (state-mismatch-handler request)

        (nil? (get-authorization-code request))
        (no-auth-code-handler request)

        :else
        (let [access-token (get-access-token provider request)]
          (-> landing-uri
            resp/redirect
            (assoc :session (-> session
                              (assoc-in [:oauth2/access-tokens id] access-token)
                              (dissoc ::authorize-state)))))))))

(defn ^:private parse-redirect-url
  [{:keys [redirect-uri]}]
  (.getPath (URI. redirect-uri)))

(defn ^:private reitit-routes-for-providers
  [provider]
  [[(:launch-uri provider) {:get {:handler (make-launch-handler provider)}}]
   [(parse-redirect-url provider) {:get {:handler (or (:redirect-handler provider)
                                                    (make-redirect-handler provider))
                                         :middleware [[wrap-params]]}}]])

(defn reitit-routes
  [providers-by-name]
  (->> providers-by-name
    (mapcat (fn [[provider-name provider-config]]
              (reitit-routes-for-providers (assoc provider-config :id provider-name))))
    (into [])))
