(ns com.vadelabs.sql-core.interface
  (:refer-clojure :exclude [first last])
  (:require
   [com.vadelabs.sql-core.ast :as ast]
   [com.vadelabs.sql-core.diff :as diff]
   [com.vadelabs.sql-core.nodes]
   [com.vadelabs.utils-core.interface :as uc]
   [honey.sql :as hsql]
   [next.jdbc :as jdbc]
   [next.jdbc.result-set :as rs]))

(defn ^:private ref?
  [{:attribute/keys [local-type]}]
  (= local-type :attribute.type/ref))

(defn eql-query
  ([attrs]
   (reduce
     (fn [acc {:attribute/keys [local-key target] :as attr}]
       (cond-> acc
         (ref? attr) (conj {local-key [target]})
         (not (ref? attr)) (conj local-key)))
     []
     attrs)))

(defn ^:private action
  [nspace entity attributes]
  (let [prefix-fn (partial uc/keywordize nspace entity)]
    [{:action/identifier (prefix-fn :insert)
      :action/entity (prefix-fn)
      :action/scope :entity
      :action/type :insert
      :action/params [:data]
      :action/output (eql-query attributes)}]))

(defn actions
  [{:keys [attributes]}]
  (->> attributes
    (uc/group-by (juxt :attribute/remote-ns :attribute/remote-entity))
    (reduce-kv
      (fn [acc [remote-ns remote-entity] attributes]
        (into acc (action remote-ns remote-entity attributes)))
      [])))

(defn create-nspace!
  [{:keys [pool nspace] :as aenv}]
  (->> [:create-schema {:if-not-exists true} nspace]
    (ast/fmt aenv)
    (jdbc/execute-one! pool)))

(defn create-schema-history!
  [{:keys [pool schema-history] :as aenv}]
  (->> [:create-table {:if-not-exists true} schema-history
        [:id {:primary-key true :not-null true :default :gen-random-uuid} :uuid]
        [:model {:not-null true} :jsonb]
        [:ddls {:not-null true} :jsonb]
        [:created-at {:default :now} :timestamp-without-time-zone]]
    (ast/fmt aenv)
    (jdbc/execute-one! pool)))

(defn latest-model!
  [{:keys [pool schema-history]}]
  (let [query (hsql/format {:select [:model]
                            :from schema-history
                            :order-by [[:created-at :desc]]
                            :limit 1})]
    (-> pool
      (jdbc/execute-one! query
        {:return-keys true
         :builder-fn rs/as-unqualified-maps})
      :model)))

(defn save-history!
  [pool schema-history model ddls]
  (let [query (hsql/format {:insert-into schema-history
                            :values [{:model [:lift model]
                                      :ddls [:lift ddls]}]})]
    (jdbc/execute-one! pool query {:return-keys true :builder-fn rs/as-unqualified-maps})))

(defn sync-schema!
  "Execute all DDL statements within a transaction"
  [{:keys [pool schema-history next-schema ddls]}]
  (jdbc/with-transaction [ds pool]
    (save-history! ds schema-history next-schema ddls)
    (doseq [ddl ddls]
      (jdbc/execute-one! ds ddl))))

(defn migration-steps
  ([next-schema]
   (migration-steps {} next-schema))
  ([previous-schema next-schema]
   (-> []
     (diff/enums previous-schema next-schema)
     (diff/tables previous-schema next-schema))))

(defn fmtv
  [aenv dsls]
  (mapv (partial ast/fmt aenv) dsls))
