(ns com.vadelabs.sql-core.mutation
  "A way to interact with stored entities"
  (:require
   [next.jdbc :as jdbc]
   [next.jdbc.transaction :as tx]
   [honey.sql :as hsql]
   [com.vadelabs.sql-core.schema :as schema]
   [com.vadelabs.sql-core.aenv :as env]
   [com.vadelabs.sql-core.coerce :as c]))

(defn ^:private success-result?
  "Asserts success back from JDBC"
  [result]
  (some-> result first :next.jdbc/update-count pos?))

(defn ^:private run-preconditions!
  "Preconditions are plain queries to run before a mutation
   to assert that the database is in the expected state."
  [jdbc mutation params pre]
  (when (seq pre)
    (run! (fn [{:keys [name query valid?]
                :or   {valid? seq}
                :as   pre}]
            (when-let [q (query params)]
              (let [result (jdbc/execute! jdbc (hsql/format q))]
                (when-not (valid? result)
                  (throw (ex-info (format "Precondition %s on mutation %s failed"
                                    name
                                    mutation)
                           {:type     :error/mutation-failed
                            :code     409
                            :mutation mutation
                            :params   (dissoc params ::metadata ::schema)
                            :pre      (dissoc pre :valid? :query)}))))))
      pre)))

(defn mutate!
  "Perform a mutation. Since mutations are spec'd,
   parameters are expected to conform it."
  ([env mutation params]
   (mutate! env mutation params {}))
  ([aenv mutation params metadata]
   (let [{:keys [spec handler pre]} (schema/resolve-mutation
                                      (env/schema aenv) mutation)
         cparams (c/write-map params)
         statement (-> cparams
                     (assoc ::schema (env/schema aenv)
                       ::metadata metadata)
                     handler
                     hsql/format)
         result (jdbc/with-transaction [tx (env/jdbc aenv)]
                  (run-preconditions! tx mutation (dissoc cparams ::schema ::metadata) pre)
                  (jdbc/execute! tx statement))]
     (when-not (success-result? result)
       (throw (ex-info (format "the mutation has failed: %s" mutation)
                {:type     :error/mutation-failed
                 :code     404 ;; Likely the mutation has failed
                          ;; because the where clauses did not match
                 :mutation mutation
                 :params   params})))
     result)))

(def schema
  "Convenience function to access the SEQL schema. To be used in mutation functions"
  ::schema)

(def metadata
  "Convenience function to access metadata provided to a mutation. To be used in mutation functions"
  ::metadata)

(defn run-in-transaction
  "Run a function of a single argument (an environment set-up to run
   in a transaction), allowing to perform isolated queries and mutations."
  [env f]
  (jdbc/with-transaction [tx (env/jdbc env)]
    (binding [tx/*nested-tx* :ignore]
      (let [env (assoc env :jdbc tx)]
        (f env)))))

(defmacro with-transaction
  "Run the forms in `fntail` in the context of a transaction against the
   provided `env` environment."
  [sym & fntail]
  `(run-in-transaction ~sym (fn [~sym] ~@fntail)))
