(ns com.vadelabs.sql-core.spec
  (:refer-clojure :exclude [type])
  (:require
   [com.vadelabs.utils-core.interface :as uc]
   [malli.core :as m]))

(def ^:private local-type->remote-type
  "Mapping from local types to SQL type names"
  {:attribute.type/bigdec :double-precision
   :attribute.type/bigint :bigint
   :attribute.type/boolean :boolean
   :attribute.type/double :double-precision
   :attribute.type/float :double-precision
   :attribute.type/datetime :timestamp-without-time-zone
   :attribute.type/keyword :text
   :attribute.type/long :bigint
   :attribute.type/map :jsonb
   :attribute.type/string :text
   :attribute.type/symbol :text
   :attribute.type/tuple :jsonb
   :attribute.type/uuid :uuid
   :attribute.type/uri :text
   :attribute.type/enum :enum})

(def ^:private column-types
  (m/schema
    (into [:enum] (vals local-type->remote-type))))

(def ^:private spec
  (m/schema
    [:schema
     {:registry
      {::spec
       [:orn
        [:create-schema [:catn
                         [:op [:enum :create-schema]]
                         [:props [:? [:map-of :keyword :any]]]
                         [:schema :keyword]]]
        [:alter-schema [:catn
                        [:op [:enum :alter-schema]]
                        [:props [:? [:map-of :keyword :any]]]
                        [:source :keyword]
                        [:target :keyword]]]
        [:drop-schema [:catn
                       [:op [:enum :drop-schema]]
                       [:props [:? [:map-of :keyword :any]]]
                       [:schema :keyword]]]
        [:create-table [:catn
                        [:op [:enum :create-table]]
                        [:props [:? [:map-of :keyword :any]]]
                        [:table :keyword]
                        [:columns [:* [:schema [:ref ::spec]]]]]]
        [:alter-table [:catn
                       [:op [:enum :alter-table]]
                       [:props [:? [:map-of :keyword :any]]]
                       [:table :keyword]
                       [:operation [:* [:schema [:ref ::spec]]]]]]
        [:drop-table [:catn
                      [:op [:enum :drop-table]]
                      [:props [:? [:map-of :keyword :any]]]
                      [:table :keyword]]]
        [:create-type [:catn
                       [:op [:enum :create-type]]
                       [:props [:? [:map-of :keyword :any]]]
                       [:type-name :keyword]
                       [:children [:* [:schema [:ref ::spec]]]]]]
        [:column [:catn
                  [:column-name :keyword]
                  [:props [:? [:map-of :keyword :any]]]
                  [:column-type [:or column-types
                                 [:tuple :keyword :int]]]]]
        [:primary-key [:catn
                       [:op [:enum :primary-key]]
                       [:props [:? [:map-of :keyword :any]]]
                       [:constraint :keyword]
                       [:children [:* [:schema [:ref ::spec]]]]]]

        [:primitive [:or :keyword :string]]

        [:select [:catn
                  [:op [:enum :select]]
                  [:props [:? [:map-of :keyword :any]]]
                  [:table :keyword]
                  [:where [:? [:or :map :keyword]]]]]
        [:insert [:catn
                  [:op [:enum :insert]]
                  [:props [:? [:map-of :keyword :any]]]
                  [:table :keyword]
                  [:children [:* :any]]]]
        [:insert-multi [:catn
                        [:op [:enum :insert-multi]]
                        [:props [:? [:map-of :keyword :any]]]
                        [:table :keyword]
                        [:cols [:vector :keyword]]
                        [:rows [:* :any]]]]]}}
     ::spec]))

(def conform
  "Conforms to hiccup data structure and creates an AST"
  (partial uc/spec-conform spec))

(comment

  (conform
    [:create-type {}
     :attribute_cardinality_enum])

  (conform
    [:create-schema {}
     :sname-one
     :sname-two
     "sthree"])
  ;; => [:create-schema {:op :create-schema, :props {}, :children [:sname-one :sname-two :sthree]}]

  (conform
    [:alter-schema {}
     :sone
     :stwo])
  ;; => [:alter-schema {:op :alter-schema, :props {}, :source :sone, :target :stwo}]

  (conform
    [:drop-schema {}
     :sone])
  ;; => [:drop-schema {:op :drop-schema, :props {}, :schema :sone}]

  (conform
    [:create-table {}
     :ntable
     [:col-one {} :uuid]])

  (conform
    [:create-type {}
     :type-name
     :eone
     :etwo
     :ethree])

  (conform
    [:create-type {}
     :tname
     [:cone :uuid]
     [:ctwo :text]])

  :rcf)

(defn ^:private schema
  [{:attribute/keys [remote-ns]}]
  [remote-ns {:name remote-ns}])

(defn ^:private schemas
  [attributes]
  (->> attributes
    (map schema)
    (into {})))

(defn ^:private primary-key
  [remote-entity attributes]
  (let [fields (->> attributes
                 (filter :attribute/primary-key)
                 (map :attribute/remote-key))]
    {:name (uc/prefix-keyword remote-entity :-pkey)
     :fields fields}))

(defn ^:private columns
  [attributes]
  (reduce
    (fn [acc {:attribute/keys [remote-key remote-type optional default]}]
      (assoc acc remote-key
        {:name remote-key
         :props (cond-> {:not-null (not optional)}
                  default (assoc :default default))
         :type remote-type}))
    {}
    attributes))

(defn ^:private table
  [remote-ns remote-entity attributes]
  {:name (->> [remote-ns remote-entity]
           (map uc/namify)
           (uc/str-join ".")
           uc/keywordize)
   :columns (columns attributes)
   :primary-key (primary-key remote-entity attributes)})

(defn ^:private tables
  [attributes]
  (let [grouped (group-by (juxt :attribute/remote-ns
                            :attribute/remote-entity)
                  attributes)]
    (reduce-kv
      (fn [acc [remote-ns remote-entity] attributes]
        (assoc acc remote-entity (table remote-ns remote-entity attributes)))
      {}
      grouped)))

(defn ^:private enums
  [attributes]
  (let [grouped (group-by :attribute/local-type attributes)
        attributes (get grouped :attribute.type/enum)]
    (reduce
      (fn [acc {:attribute/keys [remote-key remote-type options]}]
        (let [type-name (uc/prefix-keyword remote-key remote-type)]
          (assoc acc type-name
            {:name type-name
             :options options})))
      {}
      attributes)))

(defn model
  [attrs]
  {:schemas (schemas attrs)
   :tables (tables attrs)
   :enums (enums attrs)})

(defn ^:private snake-case [s]
  (uc/str-replace s #"-" "_"))

(def default-opts
  {:table-fn snake-case
   :column-fn snake-case})

(defn type
  ([local-type]
   (type {} local-type))
  ([_ local-type]
   (get local-type->remote-type local-type)))

(comment

  :rcf)
