(ns com.vadelabs.sql-core.hsql
  (:require
   [com.vadelabs.utils-exception.interface :as uex]
   [com.vadelabs.utils-str.interface :as ustr]
   [honey.sql :as hsql]))

(hsql/register-clause! :alter-column :modify-column :modify-column)

(hsql/register-fn! :quote (fn [_ [kw]]
                            [(str "\"" (name kw) "\"")]))

(hsql/register-fn! :cascade (fn [_ _]
                              ["ON DELETE CASCADE"]))

(hsql/register-fn! :no-action (fn [_ _]
                                ["ON DELETE NO ACTION"]))

(hsql/register-fn! :double-precision (fn [_ _]
                                       ["double precision"]))

(hsql/register-fn! :interval (fn [_ [field]]
                               (if (some? field)
                                 [(str "interval " field)]
                                 ["interval"])))

(hsql/register-fn! :timestamp-tz (fn [_ _]
                                   ["timestamp with time zone"]))

(hsql/register-fn! :time-tz (fn [_ _]
                              ["time with time zone"]))

(hsql/register-fn! :array (fn [_ [type]]
                            [(str type "[]")]))

(hsql/register-fn! :named-constraint (fn [_ [name constraint]]
                                       [(str "CONSTRAINT " name " " (first (hsql/format-expr constraint)))]))

(defn format-simple-expr [e context]
  (let [[sql & params] (hsql/format-expr e)]
    (when (seq params)
      (uex/raise :type uex/incorrect :code ::incorrect
        :expr e
        :params params
        :hint (str "parameters are not accepted in " context)))
    sql))

(defn ^:private format-single-column [xs]
  (ustr/join " " (cons (format-simple-expr (first xs) "column operation")
                   (map #(format-simple-expr % "column operation")
                     (rest xs)))))

(defn ^:private format-table-columns [_ xs]
  [(str "("
     (ustr/join ", " (map #'format-single-column xs))
     ")")])

(hsql/register-clause! :with-columns-raw format-table-columns nil)

(defn format-raw-update [_ x]
  [(str "UPDATE " (first (hsql/format-expr [:inline x])))])

(hsql/register-clause! :update-raw format-raw-update :update)

(defn ^:private format-add-column [_ spec]
  (if (contains? #{:if-not-exists 'if-not-exists} (last spec))
    [(str "ADD COLUMN " (hsql/sql-kw :if-not-exists) " " (format-single-column (butlast spec)))]
    [(str "ADD COLUMN " (format-single-column spec))]))

(hsql/register-clause! :add-column-raw format-add-column :add-column)

(defn ^:private format-alter-column [_ spec]
  (if (contains? #{:if-not-exists 'if-not-exists} (last spec))
    [(str "ALTER COLUMN " (hsql/sql-kw :if-not-exists) " " (format-single-column (butlast spec)))]
    [(str "ALTER COLUMN " (format-single-column spec))]))

(hsql/register-clause! :alter-column-raw format-alter-column :alter-column)

(defn ^:private create-enum [_ x]
  [(str "CREATE TYPE " x " AS ENUM")])

(hsql/register-clause! :create-enum create-enum :create-extension)

(hsql/register-clause! :create-schema (fn [_ x] (#'hsql/format-create :create :schema x nil)) nil)

(defn ^:private with-values [_ xs]
  [(str "("
     (ustr/join ", " (map #(str "'" % "'") xs))
     ")")])

(hsql/register-clause! :with-values with-values :create-extension)

(defn ^:private drop-enum [_ x]
  [(str "DROP TYPE " x)])

(hsql/register-clause! :drop-enum drop-enum :create-extension)

(defn ^:private add-enum-value [_ [enum value]]
  [(str "ALTER TYPE " enum " ADD VALUE '" value "'")])

(hsql/register-clause! :add-enum-value add-enum-value :create-extension)

(defn constraint [k xs]
  [(str (hsql/sql-kw k) " " (format-single-column xs))])

(hsql/register-clause! :add-constraint constraint :add-index)
(hsql/register-clause! :drop-constraint constraint :drop-index)
