(ns com.vadelabs.adapter-rest.interface
  (:require
   [com.vadelabs.adapter-core.interface :as ac]
   [com.vadelabs.datasource-core.interface :as dc]
   [com.vadelabs.utils-core.interface :as uc]
   [com.wsscode.pathom3.connect.operation :as pco]
   [com.vadelabs.rest-core.main :as rc]
   [clj-http.client :as http-client]
   [edn-query-language.core :as eql]))

(defmethod ac/initiate-aenv! :adapter.type/rest
  [genv {:adapter/keys [type id]}]
  (let [aenv (get-in genv [type id])
        http-client (dc/http-clent aenv)]
    (-> genv
      (assoc-in [type id :http-client] http-client))))

(defmethod ac/attributes :adapter.type/rest
  [genv {:adapter/keys [type id] :as adapter}]
  (let [aenv (get-in genv [type id])
        attributes (->> aenv dc/attributes (mapv (partial ac/prepare-attribute adapter)))
        attributes-map (ac/attributes-map attributes)]
    (-> genv
      (assoc-in [type id :attributes] attributes)
      (assoc-in [type id :attributes-map] attributes-map))))

(defmethod ac/actions :adapter.type/rest
  [genv {:adapter/keys [type id] :as adapter}]
  (let [aenv (get-in genv [type id])
        actions (->> aenv dc/actions (mapv (partial ac/prepare-action adapter)))]
    (-> genv
      (assoc-in [type id :actions] actions))))

(defn ^:private mutation
  [aenv {:action/keys [params output identifier]} mutation-fn]
  (tap> {:Params params})
  (pco/mutation
    (cond-> {::pco/op-name (uc/symbolize identifier)
             ::pco/mutate mutation-fn}
      (seq params) (assoc ::pco/params params)
      (seq output) (assoc ::pco/output output))))

(defn ^:private optional-input
  [input]
  (mapv #(pco/? %) input))

(defn ^:private resolver
  [aenv {:action/keys [identifier input params output priority optionals]} resolver-fn]
  (let [input (->> optionals optional-input (into input))]
    (pco/resolver
      (cond-> {::pco/op-name (uc/symbolize identifier)
               ::pco/resolve resolver-fn}
        priority (assoc ::pco/priority priority)
        (seq output) (assoc ::pco/output output)
        (seq input) (assoc ::pco/input input)
        (seq params) (assoc ::pco/params params)))))

(defn user-shape
  [penv]
  (get-in penv
    [:com.wsscode.pathom3.connect.planner/graph
     :com.wsscode.pathom3.connect.planner/user-request-shape]))

(defn mutation-symbol?
  [k]
  (symbol? k))

(defn eql-query
  [user-shape {:action/keys [identifier] :as action}]
  (reduce-kv
    (fn [acc k v]
      (cond
        (mutation-symbol? k) acc
        (#{identifier
           :com.wsscode.pathom3.connect.runner/attribute-errors
           :com.fulcrologic.fulcro.algorithms.form-state/config} k) acc
        (seq v) (conj acc {k (eql-query v action)})
        :else (conj acc k)))
    []
    user-shape))

(defn local-result
  [{:keys [attributes-map] :as aenv} ast-nodes result-map]
  (reduce
    (fn [acc {:keys [children dispatch-key type]}]
      (let [{:attribute/keys [remote-key target]} (get attributes-map dispatch-key)
            result-value (or (get result-map remote-key)
                           (get result-map dispatch-key))]
        (cond-> acc
          (= type :join) (assoc dispatch-key (if (map? result-value)
                                               (local-result aenv children result-value)
                                               (mapv (partial local-result aenv children) result-value)))
          (= type :prop) (assoc dispatch-key result-value))))
    {}
    ast-nodes))

(defn local-results
  [aenv eql-query results]
  (let [{:keys [children]} (eql/query->ast eql-query)]
    (if (map? results)
      (local-result aenv children results)
      (mapv (partial local-result aenv children) results))))

(defn ->remote-params
  [{:keys [attributes-map]} pparams]
  (reduce-kv
    (fn [acc k v]
      (assoc acc (get-in attributes-map [k :attribute/remote-key]) v))
    {}
    pparams))

(defn response-for
  [{:keys [http-client provider] :as aenv} {:action/keys [output authenticated] :as action} penv pparams]
  (let [pathom-query (-> penv user-shape (eql-query action))
        pathom-query (if (seq pathom-query) pathom-query output)
        oauth2-provider (-> provider namespace keyword)
        route-name (-> action :action/identifier name keyword)
        access-tokens (get-in penv [:ring/request :session :oauth2/access-tokens oauth2-provider])
        oauth2-client  (if (and authenticated (seq access-tokens))
                         (dc/oauth2-client aenv access-tokens)
                         http-client)
        pparams (->remote-params aenv pparams)
        remote-results (:body (rc/response-for oauth2-client route-name pparams))
        local-results (local-results aenv pathom-query remote-results)]
    (tap> {:access-tokens access-tokens
           :req (rc/request-for oauth2-client route-name pparams)
           :oauth2-client oauth2-client
           :route-name route-name
           :pparams pparams
           :pathom-query pathom-query
           :remote-results remote-results
           :local-results local-results
           :penv penv})
    local-results))

(defmethod ac/pathom-op [:rest :post]
  [aenv action]
  (tap> {:post action})
  (mutation aenv action (partial response-for aenv action)))

(defmethod ac/pathom-op [:rest :delete]
  [aenv action]
  (tap> {:delete action})
  (mutation aenv action (partial response-for aenv action)))

(defmethod ac/pathom-op [:rest :put]
  [aenv action]
  (tap> {:put action})
  (mutation aenv action (partial response-for aenv action)))

(defmethod ac/pathom-op [:rest :get]
  [aenv action]
  (tap> {:get action})
  (resolver aenv action (partial response-for aenv action)))

(def adapter
  {:adapter/id (uc/uuid ::adapter)
   :adapter/type :adapter.type/rest
   :adapter/nspace :rest
   :adapter/provider :rest/blank
   :adapter/qualify-attributes false
   :adapter/config {}})

(def datasources
  [{:datasource/id (uc/uuid ::datasource)
    :datasource/display-name "REST"
    :datasource/description ""
    :datasource/slug "rest"
    :datasource/icon "https://avatars.githubusercontent.com/u/80914002?s=200&v=4"
    :datasource/preview "https://xtdb.com/images/logo.svg"
    :datasource/adapter adapter
    :datasource/attributes []
    :datasource/actions []
    :datasource/categories []
    :datasource/collections []}])
