(ns com.vadelabs.adapter-postgres.interface
  (:require
   [com.vadelabs.adapter-core.interface :as ac]
   [com.vadelabs.adapter-postgres.pathom]
   [com.vadelabs.adapter-postgres.spec :as spec]
   [com.vadelabs.adapter-postgres.utils :as utils]
   [com.vadelabs.sql-core.interface :as sc]
   [com.vadelabs.utils-core.interface :as uc]
   [com.wsscode.pathom3.interface.eql :as p.eql]))

(def ^:private adapter-type :adapter.type/postgres)

(defmethod ac/initiate-aenv! adapter-type
  [genv {:adapter/keys [type id]}]
  (let [aenv (get-in genv [type id])
        datasource (sc/connect aenv)
        aenv (assoc aenv :datasource datasource)]
    (sc/create-schema aenv)
    (sc/create-migrations-table aenv)
    (-> genv
      (assoc-in [type id :datasource] datasource))))

(defmethod ac/halt-aenv! adapter-type
  [genv {:adapter/keys [type id]}]
  (let [aenv (get-in genv [type id])]
    (sc/disconnect! aenv)))

(defmethod ac/attributes adapter-type
  [genv {:adapter/keys [type nspace id attributes] :or {attributes []} :as adapter}]
  (let [attributes (mapv (fn [{:attribute/keys [remote-ns remote-entity remote-key
                                                remote-type local-key local-type] :as attribute}]
                           (ac/prepare-attribute adapter (assoc attribute
                                                           :attribute/remote-ns (or remote-ns nspace)
                                                           :attribute/remote-entity (or remote-entity (-> local-key namespace keyword))
                                                           :attribute/remote-key (or remote-key (-> local-key name keyword))
                                                           :attribute/remote-type (or remote-type (local-type spec/types-map)))))
                     attributes)
        attributes-map (ac/attributes-map attributes)
        next-sql-schema (spec/attributes->sql-schema attributes-map attributes)]
    (-> genv
      (assoc-in [type id :next-sql-schema] next-sql-schema)
      (assoc-in [type id :attributes] attributes)
      (assoc-in [type id :attributes-map] attributes-map))))

(defmethod ac/default-actions adapter-type
  [adapter attributes]
  (let [adapter-actions (->> attributes
                          utils/adapter-actions
                          (mapv (partial ac/prepare-action adapter)))
        entity-actions (->> attributes
                         utils/entity-actions
                         (mapv (partial ac/prepare-action adapter)))]
    (uc/concat-vec adapter-actions entity-actions)))

(defmethod ac/sync-schema! adapter-type
  [genv {:adapter/keys [type id]}]
  (let [aenv (get-in genv [type id])
        {prev-sql-schema :sql-schema} (sc/get-latest-sql-schema aenv)
        entity-schemas (sc/register-entities aenv)]
    (sc/migrate! (assoc aenv :prev-sql-schema prev-sql-schema))
    (-> genv
      (assoc-in [type id :prev-sql-schema] prev-sql-schema)
      (assoc-in [type id :entity-schemas] entity-schemas))))

(def adapter
  {:adapter/id (uc/uuid ::adapter)
   :adapter/nspace :pg
   :adapter/provider :postgres
   :adapter/type adapter-type
   :adapter/qualify-attributes true
   :adapter/config  {:url "jdbc:postgresql://localhost:6432/vadedb?user=vadeuser&password=vadepassword"}})

(def datasources
  [{:datasource/id (uc/uuid ::datasource)
    :datasource/display-name "PostgreSQL"
    :datasource/description "The World's Most Advanced Open Source Relational Database"
    :datasource/slug "postgresql"
    :datasource/icon "https://www.postgresql.org/media/img/about/press/elephant.png"
    :datasource/preview "https://1000logos.net/wp-content/uploads/2020/08/PostgreSQL-Logo-1024x640.png"
    :datasource/adapter adapter
    :datasource/attributes []
    :datasource/actions []
    :datasource/categories []
    :datasource/collections []}])

(comment
  (require '[com.vadelabs.datasource-studio.interface :as ds])

  (def env (ac/initiate! {:parser `parser} [(assoc adapter :adapter/attributes ds/attributes)]))

  (p.eql/process env `[(pg/save! {:data [{:profile/workspaces [{:workspace/id "110708876924555247397",
                                                                :workspace/display-name "Pragyan's Workspace",
                                                                :workspace/slug "pragyans-workspace",
                                                                :workspace/icon
                                                                "https://lh3.googleusercontent.com/a/AAcHTtdsyrS2cQKvCDtIFEZmdPuh58W5KEo1dY9XIBs=s96-c",
                                                                :workspace/owner [:profile/id "pragyan@vadelabs.com"],
                                                                :workspace/members [[:profile/id "pragyan@vadelabs.com"]]}],
                                          :profile/identities [{:identity/id "110708876924555247397",
                                                                :identity/provider :google,
                                                                :identity/provider-id "110708876924555247397",
                                                                :identity/identifier "pragyan@vadelabs.com",
                                                                :identity/profile [:profile/id "pragyan@vadelabs.com"],
                                                                :identity/response {:id "110708876924555247397",
                                                                                    :login "pragyan@vadelabs.com",
                                                                                    :name "Pragyan Tripathi",
                                                                                    :picture
                                                                                    "https://lh3.googleusercontent.com/a/AAcHTtdsyrS2cQKvCDtIFEZmdPuh58W5KEo1dY9XIBs=s96-c",
                                                                                    :source :google}}],
                                          :profile/job-title "",
                                          :profile/email "pragyan@vadelabs.com",
                                          :profile/active-workspace {:workspace/id "110708876924555247397",
                                                                     :workspace/display-name "Pragyan's Workspace",
                                                                     :workspace/slug "pragyan's-workspace",
                                                                     :workspace/icon
                                                                     "https://lh3.googleusercontent.com/a/AAcHTtdsyrS2cQKvCDtIFEZmdPuh58W5KEo1dY9XIBs=s96-c",
                                                                     :workspace/owner [:profile/id "pragyan@vadelabs.com"],
                                                                     :workspace/members [[:profile/id "pragyan@vadelabs.com"]]},
                                          :profile/id "pragyan@vadelabs.com",
                                          :profile/display-name "Pragyan Tripathi",
                                          :profile/settings [{:setting/id "pragyan@vadelabs.com:notify-on-email",
                                                              :setting/option :notify-on-email,
                                                              :setting/type :boolean,
                                                              :setting/value true,
                                                              :setting/description "Send me email notifications"}
                                                             {:setting/id "pragyan@vadelabs.com:notify-about-everything",
                                                              :setting/option :notify-about-everything,
                                                              :setting/type :boolean,
                                                              :setting/value true,
                                                              :setting/description "Notify me about everything"}
                                                             {:setting/id "pragyan@vadelabs.com:notify-about-mentions",
                                                              :setting/option :notify-about-mentions,
                                                              :setting/type :boolean,
                                                              :setting/value true,
                                                              :setting/description
                                                              "Only notify me when someone sends me a Ping or @mentions me"}
                                                             {:setting/id "pragyan@vadelabs.com:notify-deployment-failures",
                                                              :setting/option :notify-deployment-failures,
                                                              :setting/type :boolean,
                                                              :setting/value true,
                                                              :setting/description "Notify about deployment failures"}
                                                             {:setting/id "pragyan@vadelabs.com:notify-integration-updates",
                                                              :setting/option :notify-integration-updates,
                                                              :setting/type :boolean,
                                                              :setting/value true,
                                                              :setting/description "Notify about integration updates"}
                                                             {:setting/id "pragyan@vadelabs.com:notify-warnings",
                                                              :setting/option :notify-warnings,
                                                              :setting/type :boolean,
                                                              :setting/value true,
                                                              :setting/description "Notify about warnings"}],
                                          :profile/avatar "https://lh3.googleusercontent.com/a/AAcHTtdsyrS2cQKvCDtIFEZmdPuh58W5KEo1dY9XIBs=s96-c",
                                          :profile/tokens []}]})])

  (p.eql/process env
    [{[:pg.profile/id #uuid "bead2ebf-3299-53aa-a07d-87bc386608ab"]
      [:pg.profile/id
       :pg.profile/display-name
       {:pg.profile/active-workspace [:pg.workspace/id
                                      :pg.workspace/display-name
                                      :pg.workspace/created-at]}]}])

  :rcf)
