(ns com.vadelabs.adapter-postgres.spec
  (:require
   [com.vadelabs.utils-core.interface :as uc]
   [com.vadelabs.utils-core.string :as ustr]))

(def types-map
  {:attribute.type/bigdec :double
   :attribute.type/bigint :bigint
   :attribute.type/boolean :boolean
   :attribute.type/double :double
   :attribute.type/enum :enum
   :attribute.type/float :double
   :attribute.type/secret :string
   :attribute.type/datetime :timestamp-tz
   :attribute.type/keyword :string
   :attribute.type/long :bigint
   :attribute.type/map :jsonb
   :attribute.type/ref :ref
   :attribute.type/string :string
   :attribute.type/symbol :string
   :attribute.type/tuple :array
   :attribute.type/uuid :uuid
   :attribute.type/uri :string})

(defn ^:private attribute->schema
  [{:attribute/keys [remote-ns]}]
  [remote-ns {:name remote-ns}])

(defn ^:private ->schemas
  [attributes]
  (->> attributes
    (map attribute->schema)
    (into {})))

(defn ^:private ->enums
  [attributes]
  (->> attributes
    (reduce (fn [acc {:attribute/keys [remote-ns values options] :as attribute}]
              (let [type-name (->> attribute
                                ((juxt :attribute/remote-entity :attribute/remote-key :attribute/remote-type))
                                (map uc/namify)
                                (ustr/join "_")
                                (uc/prefix-keyword "." remote-ns))]
                (assoc acc type-name {:name type-name :values (or values options)})))
      {})))

(defn ^:private ->primary-key
  [remote-entity attributes]
  (let [fields (->> attributes
                 (filter :attribute/primary-key)
                 (mapv :attribute/remote-key))]
    {:name (->> [remote-entity :pkey]
             (map uc/namify)
             (ustr/join "_")
             uc/keywordize)
     :fields fields}))

(defn ^:private ->columns
  [attributes-map attributes]
  (reduce
    (fn [acc {:attribute/keys [cardinality remote-key remote-ns remote-type remote-entity target optional default values options]}]
      (let [target-type (when (= remote-type :ref)
                          (get-in attributes-map [target :attribute/remote-type]))
            enum-name (when (= remote-type :enum)
                        (uc/prefix-keyword "." remote-ns (->> [remote-entity remote-key remote-type]
                                                           (map uc/namify)
                                                           (ustr/join "_"))))
            table-name (uc/prefix-keyword "." remote-ns remote-entity)
            column-type (if (= cardinality :many) :array
                          (or target-type remote-type))]
        (assoc acc remote-key {:value (cond-> {:type column-type}
                                        (= column-type :enum) (assoc :values (mapv name (or values options))))
                               :properties (cond-> {:name remote-key :table table-name}
                                             (= column-type :enum) (assoc :type-name enum-name)
                                             (= column-type :array) (assoc :of remote-type)
                                             optional (assoc :optional optional)
                                             default (assoc :default default))})))
    {}
    attributes))

(defn ^:private ->table
  [attributes-map remote-ns remote-entity attributes]
  {:type :map
   :properties {:name (uc/prefix-keyword "." remote-ns remote-entity)
                :primary-key (->primary-key remote-entity attributes)}
   :keys (->columns attributes-map attributes)})

(defn ->tables
  [attributes-map attributes]
  (let [grouped (uc/group-by (juxt :attribute/remote-ns :attribute/remote-entity) attributes)]
    (reduce-kv
      (fn [acc [remote-ns remote-entity] attributes]
        (assoc acc (uc/prefix-keyword "." remote-ns remote-entity) (->table attributes-map remote-ns remote-entity attributes)))
      {}
      grouped)))

;; Foreign Keys
(defn ->foreign-keys
  [attributes-map ref-attrs]
  (->> ref-attrs
    (reduce (fn [acc {:attribute/keys [remote-ns remote-entity remote-key target]}]
              (let [{target-entity :attribute/remote-entity
                     target-key :attribute/remote-key
                     target-type :attribute/remote-type
                     target-ns :attribute/remote-ns} (target attributes-map)
                    constraint-name (->> [remote-ns remote-entity remote-key :fkey]
                                      (map uc/namify)
                                      (ustr/join "_")
                                      uc/keywordize)]
                (assoc acc constraint-name {:name constraint-name
                                            :source-table (uc/prefix-keyword "." remote-ns remote-entity)
                                            :source-column remote-key
                                            :source-column-type target-type
                                            :target-table (uc/prefix-keyword "." target-ns target-entity)
                                            :target-column target-key
                                            :on-delete :cascade
                                            :on-update :cascade})))
      {})))

(defn ref-attrs
  [attributes]
  (filter :attribute/target attributes))

(defn unref-attrs
  [attributes]
  (filter (complement :attribute/target) attributes))

(defn enum-attrs
  [attributes]
  (filter (fn [{:attribute/keys [local-type]}]
            (= local-type :attribute.type/enum))
    attributes))

(defn ->join-attribute-pair
  [attributes-map attribute]
  (let [{source-id :attribute/id
         source-description :attribute/description
         source-group :attribute/group
         source-entity :attribute/remote-entity
         source-key :attribute/remote-key
         source-target :attribute/target} attribute
        {target-id :attribute/id
         target-description :attribute/description
         target-group :attribute/group
         target-ns :attribute/remote-ns
         target-local-key :attribute/local-key} (source-target attributes-map)
        group (uc/prefix-keyword "-" source-group target-group)
        entity (uc/prefix-keyword "-" source-entity source-key)]
    [{:attribute/id source-id
      :attribute/description source-description
      :attribute/group group
      :attribute/local-key (uc/keywordize target-ns entity source-entity)
      :attribute/local-type :attribute.type/ref
      :attribute/target (uc/keywordize source-entity :id)
      :attribute/remote-entity entity
      :attribute/remote-key source-entity
      :attribute/remote-type :ref
      :attribute/primary-key true
      :attribute/optional true ;; TODO: be more precise. This will enable deletion with setting null
      :attribute/remote-ns target-ns
      :attribute/cardinality :one}
     {:attribute/id target-id
      :attribute/description target-description
      :attribute/group group
      :attribute/local-key (uc/keywordize target-ns entity source-key)
      :attribute/local-type :attribute.type/ref
      :attribute/target target-local-key
      :attribute/remote-entity entity
      :attribute/remote-key source-key
      :attribute/remote-type :ref
      :attribute/primary-key true
      :attribute/optional true
      :attribute/remote-ns target-ns
      :attribute/cardinality :one}]))

(defn ->join-attrs
  [attributes-map ref-attrs]
  (->> ref-attrs
    (mapcat (partial ->join-attribute-pair attributes-map))))

(defn attributes->sql-schema
  [attributes-map attributes]
  (let [[enum-attrs ref-attrs unref-attrs] ((juxt enum-attrs ref-attrs unref-attrs) attributes)
        join-attrs (->join-attrs attributes-map ref-attrs)
        all-attributes (into unref-attrs join-attrs)]
    {:enums (->enums enum-attrs)
     :schemas (->schemas attributes)
     :tables  (->tables attributes-map all-attributes)
     :foreign-keys (->foreign-keys attributes-map join-attrs)}))

(defn attributes->field-schema
  [attributes]
  (reduce
    (fn [acc {:attribute/keys [remote-key remote-type cardinality]}]
      (let [remote-type (if (= cardinality :many)
                          [:vector remote-type]
                          remote-type)]
        (conj acc [remote-key {} remote-type])))
    [:map {}]
    attributes))

(defn attributes->entity-schema
  [attributes]
  (->> attributes
    (uc/group-by (juxt :attribute/remote-ns :attribute/remote-entity))
    (reduce-kv (fn [acc [remote-ns remote-entity] attributes]
                 (assoc acc (uc/prefix-keyword "." remote-ns remote-entity) (attributes->field-schema attributes)))
      {})))
