(ns com.vadelabs.datasource-core.interface
  (:require
   [camel-snake-kebab.core :as csk]
   [com.vadelabs.adapter-core.interface :as ac]
   [com.vadelabs.datasource-core.interceptors :as dci]
   [com.vadelabs.utils-core.interface :as uc]
   [com.vadelabs.utils-exception.interface :as uex]
   [com.vadelabs.utils-str.interface :as ustr]))

(def common-interceptors
  [dci/keywordize-params
   dci/set-method
   dci/set-url
   dci/set-query-params
   dci/set-body-params
   dci/set-form-params
   dci/set-header-params
   dci/enqueue-route-interceptors
   dci/encode-request-body
   dci/encode-response-body])

(def perform-sync-request
  (conj common-interceptors dci/perform-sync-request))

(def perform-sse-request
  (conj common-interceptors dci/perform-sse-request))

(def perform-async-request
  (conj common-interceptors dci/perform-async-request))

(defmulti attributes
  :provider)

(defmulti actions
  :provider)

(defmulti http-clent
  "Multimethod to instantiate rest datasource"
  :provider)

(defmulti oauth2-client
  (fn [{:keys [provider]} _access-tokens]
    provider))

(defmulti oauth2-config
  :provider)

(defmethod http-clent :default
  [{:keys [identifier] :as aenv}]
  (uex/raise :type :not-implemented
    :code :http-instance-not-implemented
    :aenv aenv
    :hint (ustr/format "No http instance available for %s" identifier)))

(defmethod oauth2-client :default
  [{:keys [identifier] :as aenv} & _]
  (uex/raise :type :not-implemented
    :code :oauth2-instance-not-implemented
    :aenv aenv
    :hint (ustr/format "No oauth2 instance available for %s" identifier)))

(defmethod oauth2-config :default
  [{:keys [identifier] :as aenv}]
  (uex/raise :type :not-implemented
    :code :oauth2-config-not-implemented
    :aenv aenv
    :hint (ustr/format "No oauth2 config available for %s" identifier)))

(defn ^:private handler->attributes
  [{:keys [query-schema response-schema]}]
  (cond-> []
    (seq response-schema) (into (->> response-schema
                                  vals
                                  (filter sequential?)
                                  (mapcat ac/malli-schema->attributes)))
    query-schema (into (->> query-schema ac/malli-schema->attributes))))

(defn handlers->attributes
  [handlers]
  (into [] (mapcat handler->attributes handlers)))

(defn ^:private handler->action
  [{:keys [info method route-name query-schema body-schema authenticated path-schema response-schema]}]
  (let [service-name (:x-serviceName info)
        query? (#{:get} method)
        mutation? (not query?)
        identifier (if service-name
                     (uc/keywordize (csk/->kebab-case-keyword service-name) route-name)
                     route-name)
        schemas (cond-> {}
                  query-schema (assoc :query-schema query-schema)
                  path-schema (assoc :path-schema path-schema)
                  body-schema (assoc :body-schema body-schema))
        input (->> schemas
                handler->attributes
                ac/attributes->pathom-query)]
    (cond-> {:action/id (uc/uuid identifier)
             :action/identifier identifier
             :action/display-name (uc/namify identifier)
             :action/type method
             :action/deprecated false
             :action/authenticated authenticated
             :action/output (->> {:response-schema response-schema}
                              handler->attributes
                              ac/attributes->pathom-query)}
      query? (assoc :action/input [])
      mutation? (assoc :action/params input))))

(defn handlers->actions
  [handlers]
  (mapv handler->action handlers))
