(ns com.vadelabs.rest-core.context
  (:require
   [com.vadelabs.utils-core.interface :as uc]
   #?@(:clj [[clojure.core.async :refer [go <!]]
             [clojure.core.async.impl.protocols :as asyncp]]
       :cljs [[cljs.core.async :refer [<!] :refer-macros [go]]
              [cljs.core.async.impl.protocols :as asyncp]])))

(def queue (uc/queue))

;; TODO: liter this through the call sites below.  This will allow pattern match on the results
(defn ^:private exception->ex-info [exception execution-id interceptor stage]
  (ex-info (str "Interceptor Exception: " #?(:clj  (.getMessage exception)
                                             :cljs (.-message exception)))
    (merge {:execution-id execution-id
            :stage        stage
            :interceptor  (:name interceptor)
            :type         (type exception)
            :exception    exception}
      (ex-data exception))
    exception))

(defn ^:private try-f
  "If f is not nil, invokes it on context. If f throws an exception,
  assoc's it on to context as ::error."
  [context interceptor stage]
  (let [execution-id (::execution-id context)]
    (if-let [f (get interceptor stage)]
      (try
        (f context)
        (catch #?(:clj Throwable :cljs js/Object) e
          (assoc context ::error (exception->ex-info e execution-id interceptor stage))))
      context)))

(defn ^:private try-error
  "If error-fn is not nil, invokes it on context and the current ::error
  from context."
  [context interceptor]
  (let [execution-id (::execution-id context)]
    (if-let [error-fn (get interceptor :error)]
      (let [ex (::error context)]
        (try (error-fn (dissoc context ::error) ex)
          (catch #?(:clj Throwable :cljs js/Object) e
            (if (identical? (type e) (type (:exception ex)))
              context
              (-> context
                (assoc ::error (exception->ex-info e execution-id interceptor :error))
                (update-in [::suppressed] conj ex))))))
      context)))

(defn ^:private check-terminators
  "Invokes each predicate in ::terminators on context. If any predicate
  returns true, removes ::queue from context."
  [context]
  (if (some #(% context) (::terminators context))
    (let [execution-id (::execution-id context)]
      (dissoc context ::queue))
    context))

#?(:cljs
   (defn with-bindings [_ res] res))

(defn channel? [x]
  (satisfies? asyncp/ReadPort x))

(declare terminate)
(declare execute)

(defn go-async [old-context context-ch]
  (go (execute (<! context-ch)))
  (terminate old-context))

(defn ^:private enter-all-with-binding
  "Invokes :enter functions of all Interceptors on the execution
  ::queue of context, saves them on the ::stack of context. Returns
  updated context."
  [context]
  #_(uc/debug :in 'enter-all :execution-id (::execution-id context))
  (loop [context context]
    (let [queue (::queue context)
          stack (::stack context)]
      #_(uc/trace :context context)
      (if (empty? queue)
        context
        (let [interceptor (peek queue)
              pre-bindings (:bindings context)
              old-context context
              context (-> context
                        (assoc ::queue (pop queue))
                          ;; conj on nil returns a list, acts like a stack:
                        (assoc ::stack (conj stack interceptor))
                        (try-f interceptor :enter))]
          (cond
            (channel? context) (go-async old-context context)
            (::error context) (dissoc context ::queue)
            (not= (:bindings context) pre-bindings) (assoc context ::rebind true)
            true (recur (check-terminators context))))))))

(defn ^:private enter-all
  "Establish the bindings present in `context` as thread local
  bindings, and then invoke enter-all-with-binding. Conditionally
  re-establish bindings if a change in bindings is made by an
  interceptor."
  [context]
  (let [context (with-bindings (:bindings context {})
                  (enter-all-with-binding context))]
    (if (::rebind context)
      (recur (dissoc context ::rebind))
      context)))

(defn ^:private leave-all-with-binding
  "Unwinds the context by invoking :leave functions of Interceptors on
  the ::stack of context. Returns updated context."
  [context]
  #_(uc/debug :in 'leave-all :execution-id (::execution-id context))
  (loop [context context]
    (let [stack (::stack context)]
      #_(uc/trace :context context)
      (if (empty? stack)
        context
        (let [interceptor (peek stack)
              pre-bindings (:bindings context)
              old-context context
              context (assoc context ::stack (pop stack))
              context (if (::error context)
                        (try-error context interceptor)
                        (try-f context interceptor :leave))]
          (cond
            (channel? context) (go-async old-context context)
            (not= (:bindings context) pre-bindings) (assoc context ::rebind true)
            :else (recur context)))))))

(defn leave-all
  "Establish the bindings present in `context` as thread local
  bindings, and then invoke leave-all-with-binding. Conditionally
  re-establish bindings if a change in bindings is made by an
  interceptor."
  [context]
  (let [context (with-bindings (:bindings context {})
                  (leave-all-with-binding context))]
    (if (::rebind context)
      (recur (dissoc context ::rebind))
      context)))

(defn enqueue
  "Adds interceptors to the end of context's execution queue. Creates
  the queue if necessary. Returns updated context."
  [context & interceptors]
  #_(uc/trace :enqueue (map :name interceptors) :context context)
  (update-in context [::queue]
    (fnil into queue)
    interceptors))

(defn enqueue*
  "Like 'enqueue' but the second argument is a sequence of interceptors
  to add to the context's execution queue."
  [context interceptors]
  (apply enqueue context interceptors))

(defn terminate
  "Removes all remaining interceptors from context's execution queue.
  This effectively short-circuits execution of Interceptors' :enter
  functions and begins executing the :leave functions."
  [context]
  #_(uc/trace :in 'terminate :context context)
  (dissoc context ::queue))

(defn terminate-when
  "Adds pred as a terminating condition of the context. pred is a
  function that takes a context as its argument. It will be invoked
  after every Interceptor's :enter function. If pred returns logical
  true, execution will stop at that Interceptor."
  [context pred]
  (update-in context [::terminators] conj pred))

(defn ^:private begin [context]
  (if (contains? context ::execution-id)
    context
    (let [execution-id (uc/uuid)]
      #_(ulog/debug :begin/executing :execution-id execution-id)
      #_(uc/trace :context context)
      (assoc context ::execution-id execution-id))))

(defn ^:private end [context]
  #_(ulog/debug :end/executing :c)
  #_(uc/trace :context context)
  context)

(defn execute [context]
  (let [context (some-> context
                  begin
                  enter-all
                  (dissoc ::queue)
                  leave-all
                  (dissoc ::stack ::execution-id)
                  end)]
    (if-let [ex (::error context)]
      (throw ex)
      context)))

(defn remove-stack [ctx]
  (-> ctx terminate (dissoc ::stack)))

(defn enqueue-route-interceptors
  [ctx interceptors]
  (update ctx ::queue #(into (into queue interceptors) %)))
