(ns com.vadelabs.sql-core.ddl
  (:require
   [clojure.core.match :refer [match]]
   [com.vadelabs.sql-core.hsql]
   [com.vadelabs.utils-core.interface :as uc]
   [com.vadelabs.utils-core.string :as ustr]
   [com.vadelabs.utils-exception.interface :as uex]))

;; ENUMS ;;

(defn ^:private create-enum
  "Returns HoneySQL formatted map representing create enum SQL clause"
  [{:keys [name values]}]
  {:create-enum name :with-values values})
(defn create-enums
  [created _prev-sql-schema next-sql-schema]
  (let [enums (if (vector? created) created [created])]
    (->> enums
      (sort-by name)
      (mapv #(create-enum (% next-sql-schema))))))

(defn drop-enums
  [dropped prev-sql-schema _next-sql-schema])

(defn alter-enums
  [altered prev-sql-schema next-sql-schema])

;; COLUMNS ;;

(defn ->array-type
  "Returns supported by Postgres array type"
  [type]
  (case type
    (:smallint) "int2"
    (:int :integer 'int? 'integer?) "int4"
    (:bigint) "int8"
    (:real 'float?) "float4"
    (:double 'double? :decimal :numeric 'number?) "float8"
    (:boolean 'boolean?) "bool"
    (:char 'char?) "varchar"
    (:string 'string?) "text"
    (:enum) "text" ;; TODO: figure out a way to use enum datatype instead of hard-coding text
    (uex/raise :type uex/unsupported
      :code :unsupported-type
      :ctx {:type type}
      :hint (ustr/format "Array type doesn't support %s" type))))
(defn column-sql-schema->column-type
  "Given a field type and optional properties will return a unified (simplified) type representation.
   Few implementation notes:
   interval with fields - only DAY, HOUR, MINUTE and SECOND supported due to conversion from Duration class
   time with timezone doesn't really work - zone part is missed somewhere in between DB and PGDriver"
  [{:keys [properties value]}]
  (let [column-type (:type value)]
    (match [column-type properties]
    ;; uuid
      [(:or :uuid 'uuid?) _] :uuid

    ;; numeric
      [:smallint _] :smallint
      [:bigint _] :bigint
      [(:or :int :integer 'int? 'integer?) _] :integer
      [(:or :decimal :numeric 'number?) {:precision p :scale s}] [:numeric p s]
      [(:or :decimal :numeric 'number?) {:precision p}] [:numeric p]
      [(:or :decimal :numeric 'number?) _] :numeric
      [(:or :real 'float?) _] :real
      [(:or :double 'double?) _] [:double-precision]

      [:smallserial {:as-reference true}] :smallint
      [:serial {:as-reference true}] :integer
      [:bigserial {:as-reference true}] :bigint
      [:smallserial _] :smallserial
      [:serial _] :serial
      [:bigserial _] :bigserial

    ;; character
      [(:or :char 'char?) {:max max}] [:char max]
      [(:or :string 'string?) {:max max}] [:varchar max]
      [(:or :string 'string?) _] :text

      [(:or :boolean 'boolean?) _] :boolean
      [:enum {:type-name enum-name}] [:inline (keyword enum-name)]
      [:jsonb _] :jsonb
      [:array {:of atype}] [:array (->array-type atype)]

      [:timestamp _] :timestamp
      [:timestamp-tz _] [:timestamp-tz]
      [:date _] :date
      [:time _] :time
      [:time-tz _] [:time-tz]
      [:interval {:fields fields}] [:interval fields]
      [:interval _] :interval

      :else (throw (ex-info (str "Unknown type for column " column-type) {:column-type column-type})))))

(defn ->default-value
  [default-value]
  (if (keyword? default-value)
    (uc/namify default-value)
    default-value))

(defn column-sql-schema->column-modifiers
  [{:keys [properties]}]
  (let [{:keys [optional default]} properties]
    (cond-> []
      (not optional) (conj [:not nil])
      (some? default) (conj [:default (->default-value default)]))))

(def ^:private reserved
  #{:group :cardinality})

(defn safeguard
  [column-keyword]
  (if (contains? reserved column-keyword)
    [:quote column-keyword]
    column-keyword))

(defn ^:private ->column
  [[column-keyword column-schema]]
  (into [(safeguard column-keyword) (column-sql-schema->column-type column-schema)]
    (column-sql-schema->column-modifiers column-schema)))

(defn ^:private ->table-constraints
  [columns {:keys [name fields]}]
  (cond-> columns
    (seq fields) (conj [[:named-constraint name (into [:primary-key] fields)]])))

;; TABLES ;;
(defn ^:private create-table
  "Returns HoneySQL formatted map representing create SQL clause"
  [table columns primary-key]
  (let [columns (->table-constraints columns primary-key)]
    {:create-table table :with-columns columns}))
(defn create-tables
  [created _prev-sql-schema next-sql-schema]
  (let [tables (if (vector? created) created [created])]
    (->> tables
      (sort-by name)
      (mapv (fn [table]
              (let [columns (-> next-sql-schema table :keys)
                    table-name (-> next-sql-schema table :properties :name)
                    primary-key (-> next-sql-schema table :properties :primary-key)]
                (create-table table-name (mapv ->column columns) primary-key)))))))

(defn drop-tables
  [dropped prev-sql-schema _next-sql-schema])

(defn alter-tables
  [altered prev-sql-schema next-sql-schema])

;; FOREIGN KEY CONSTRAINT ;;
(defn ^:private create-foreign-key
  [{:keys [name source-table source-column target-table
           target-column on-delete on-update]}]
  {:alter-table [source-table {:add-constraint [name
                                                [:foreign-key source-column]
                                                [:references target-table target-column]
                                                (when on-delete [:on-delete on-delete])
                                                (when on-update [:on-update on-update])]}]})

(defn create-foreign-keys
  [created _prev-sql-schema next-sql-schema]
  (let [foreign-keys (if (vector? created) created [created])]
    (->> foreign-keys
      (sort-by name)
      (mapv #(create-foreign-key (% next-sql-schema))))))

(defn drop-foreign-keys
  [dropped prev-sql-schema next-sql-schema])

(defn alter-foreign-keys
  [altered prev-sql-schema next-sql-schema])
