(ns com.vadelabs.sql-core.introspection
  (:require
   [clojure.datafy :as df]
   [com.vadelabs.utils-core.interface :as uc]
   [com.vadelabs.utils-core.string :as ustr]
   [next.jdbc :as jdbc]
   [next.jdbc.datafy :as njdf]
   [next.jdbc.result-set :as njrs]
   [next.jdbc.result-set :as rs]
   [camel-snake-kebab.core :as csk])
  (:import
   [javax.sql DataSource]
   [java.sql DatabaseMetaData Connection]
   [java.time Clock]
   [org.postgresql.util PSQLException]))

(def default-adapter
  {:schema-pattern nil
   :column-types   {#"int" :integer}
   :predicates     {:nullable?      (fn [{:keys [is_nullable] :as _raw}]
                                      (= "YES" is_nullable))
                    :unique?        (fn [{:keys [indexes] :as _raw}]
                                      (->> indexes
                                        (filter (complement :non_unique))
                                        seq))
                    :autoincrement? (fn [{:keys [is_autoincrement] :as _raw}]
                                      (= "YES" is_autoincrement))}})

(defmulti adapter* :dbtype)

(defmethod adapter* :postgresql
  [{:keys [nspace]}]
  {:schema-pattern (uc/namify (or nspace :public))
   :column-types   {:varchar :string
                    :timetz :time-tz
                    :int2 :smallint
                    :int4 :integer
                    :int8 :bigint}})

(defmethod adapter* :default
  [_]
  default-adapter)

(defn adapter
  [db-info]
  (let [{:keys [schema-pattern column-types predicates]} (adapter* db-info)]
    {:schema-pattern (or schema-pattern (:schema-pattern default-adapter))
     :column-types   (merge (:column-types default-adapter) column-types)
     :predicates     (merge (:predicates default-adapter) predicates)}))

(defmulti ^:private db-config :provider)

(defmethod db-config :postgres
  [{:keys [nspace] :or {nspace :public}}]
  {:schema-pattern (uc/namify nspace)
   :column-types {:varchar :string
                  :timetz :time-tz
                  :int2 :smallint
                  :int4 :integer
                  :int8 :bigint}})

(defn ^:private product-name
  [metadata]
  (-> metadata
    .getDatabaseProductName
    ustr/lower
    uc/keywordize))

(defn ^:private catalog
  [metadata]
  (-> metadata .getConnection .getCatalog))

(defn get-columns
  ([aenv]
   (get-columns aenv nil))
  ([{:keys [datasource metadata db-config catalog]} table]
   (-> metadata
     (.getColumns catalog (:schema-pattern db-config) table nil)
     (rs/datafiable-result-set datasource {:builder-fn rs/as-unqualified-kebab-maps}))))

(defn get-tables
  [{:keys [datasource metadata db-config catalog]}]
  (-> metadata
    (.getTables catalog (:schema-pattern db-config) nil (into-array ["TABLE"]))
    (rs/datafiable-result-set datasource {:builder-fn rs/as-unqualified-kebab-maps})))

(defn prepare
  [{:keys [connection] :as aenv}]
  (let [^DatabaseMetaData metadata (.getMetaData connection)
        dbtype (product-name metadata)
        catalog (catalog metadata)
        db-config (db-config aenv)]
    (assoc aenv :metadata metadata
      :dbtype dbtype
      :catalog catalog
      :db-config db-config)))

(defn get-primary-keys
  [{:keys [datasource metadata catalog]} table-schema table-name]
  (-> metadata
    (.getPrimaryKeys catalog table-schema table-name)
    (rs/datafiable-result-set datasource {:builder-fn rs/as-unqualified-kebab-maps})
    ((partial uc/index-by #(-> % :column-name csk/->kebab-case-keyword)))))

(defn get-foreign-keys
  [{:keys [datasource metadata catalog]} table-schema table-name]
  (-> metadata
    (.getImportedKeys catalog table-schema table-name)
    (rs/datafiable-result-set datasource {:builder-fn rs/as-unqualified-kebab-maps})
    ((partial uc/index-by #(-> % :fkcolumn-name csk/->kebab-case-keyword)))))

(defn get-index-info
  [{:keys [datasource metadata catalog]} table-schema table-name]
  (-> metadata
    (.getIndexInfo catalog table-schema table-name true false)
    (rs/datafiable-result-set datasource {:builder-fn rs/as-unqualified-kebab-maps})
    ((partial uc/index-by #(-> % :column-name csk/->kebab-case-keyword)))))

(defn column-keyword
  [column-name]
  (-> column-name
    csk/->kebab-case-keyword))

(defn column-type-keyword
  [{:keys [db-config]} type-name]
  (let [column-types (:column-types db-config)
        type-keyword (uc/keywordize type-name)]
    (get column-types type-keyword type-keyword)))

(def ^:private default-column-size 2147483647)

(defn reference-target
  [fk-reference]
  (->> fk-reference
    ((juxt :pktable-schem :pktable-name :pkcolumn-name))
    (mapv keyword)))

(defn ->malli-schema
  [aenv table-schema table-name columns]
  (let [primary-keys (get-primary-keys aenv table-schema table-name)
        foreign-keys (get-foreign-keys aenv table-schema table-name)
        index-info (get-index-info aenv table-schema table-name)
        table-key (uc/prefix-keyword "." table-schema table-name)
        composite-primary-key? (> (-> primary-keys keys count) 1)
        simple-primary-key? #(and (not composite-primary-key?) (get primary-keys %))
        table-props (cond-> {:table table-key}
                      composite-primary-key? (assoc :primary-key (-> primary-keys keys vec)))]
    (tap> {:pg primary-keys
           :fk foreign-keys
           :ii index-info
           :cols columns})
    (reduce (fn [acc {:keys [column-name nullable type-name column-size]}]
              (let [column-key (column-keyword column-name)
                    column-type (column-type-keyword aenv type-name)
                    column-type (if (and (= column-type :string) (not= column-size default-column-size))
                                  [column-type {:max column-size}]
                                  column-type)
                    foreign-key-reference (get foreign-keys column-key)
                    props (cond-> {}
                            (= nullable 1) (assoc :optional true)
                            (simple-primary-key? column-key) (assoc :primary-key true)
                            foreign-key-reference (assoc :reference-target (reference-target foreign-key-reference)))]
                (conj acc [column-key props column-type])))
      [:spec table-props]
      columns)))

(defn execute!
  [aenv]
  (let [db-info (prepare aenv)
        columns (->> db-info
                  get-columns
                  (group-by (juxt :table-schem :table-name)))]
    (reduce-kv (fn [acc [table-schema table-name] columns]
                 (assoc acc (uc/keywordize table-schema table-name) (->malli-schema db-info table-schema table-name columns)))
      {}
      columns)))

(comment

  (def ds (jdbc/get-datasource {:jdbcUrl "jdbc:postgresql://localhost:6432/vadedb?user=vadeuser&password=vadepassword"}))

  (execute! {:datasource ds
             :connection (jdbc/get-connection ds)
             :nspace :pg
             :provider :postgres})

  :rcf)
