(ns com.vadelabs.sql-core.migration
  (:require
   [com.vadelabs.sql-core.ddl :as ddl]
   [com.vadelabs.sql-core.entity :as entity]
   [com.vadelabs.sql-core.types]
   [com.vadelabs.utils-core.interface :as uc]
   [com.vadelabs.utils-str.interface :as ustr]
   [honey.sql :as hsql]
   [next.jdbc :as jdbc]
   [next.jdbc.sql.builder :as builder]
   [next.jdbc.result-set :as jdbc.rs])
  (:import
   [javax.sql DataSource]
   [java.sql DatabaseMetaData Connection]
   [java.time Clock]
   [org.postgresql.util PSQLException]))

(defn ^:private ->changes
  [prev-sql-schema next-sql-schema]
  (let [changes (uc/changes prev-sql-schema next-sql-schema)
        created (uc/created changes prev-sql-schema next-sql-schema)
        dropped (uc/dropped changes prev-sql-schema next-sql-schema)
        altered (uc/altered changes prev-sql-schema next-sql-schema)]
    {:created created :dropped dropped :altered altered}))

(defn ->enum-changes
  [rs {prev-sql-schema :enums} {next-sql-schema :enums}]
  (let [{:keys [created dropped altered]} (->changes prev-sql-schema next-sql-schema)]
    (cond-> rs
      (seq created) (into (ddl/create-enums created prev-sql-schema next-sql-schema))
      (seq dropped) (into (ddl/drop-enums dropped prev-sql-schema next-sql-schema))
      (seq altered) (into (ddl/alter-enums altered prev-sql-schema next-sql-schema)))))

(defn ->table-changes
  [rs {prev-sql-schema :tables} {next-sql-schema :tables}]
  (let [{:keys [created dropped altered]} (->changes prev-sql-schema next-sql-schema)]
    (cond-> rs
      (seq created) (into (ddl/create-tables created prev-sql-schema next-sql-schema))
      (seq dropped) (into (ddl/drop-tables dropped prev-sql-schema next-sql-schema))
      (seq altered) (into (ddl/alter-tables altered prev-sql-schema next-sql-schema)))))

(defn ->foreign-key-changes
  [rs {prev-sql-schema :foreign-keys} {next-sql-schema :foreign-keys}]
  (let [{:keys [created dropped altered]} (->changes prev-sql-schema next-sql-schema)]
    (cond-> rs
      (seq created) (into (ddl/create-foreign-keys created prev-sql-schema next-sql-schema))
      (seq dropped) (into (ddl/drop-foreign-keys dropped prev-sql-schema next-sql-schema))
      (seq altered) (into (ddl/alter-foreign-keys altered prev-sql-schema next-sql-schema)))))

(defn hsql-statements
  ([sql-schema]
   (hsql-statements {} sql-schema))
  ([prev-sql-schema  next-sql-schema]
   (-> []
     (->enum-changes prev-sql-schema next-sql-schema)
     (->table-changes prev-sql-schema next-sql-schema)
     (->foreign-key-changes prev-sql-schema next-sql-schema))))

(defn ->sql-statements
  [hsql-statements]
  (mapv hsql/format hsql-statements))

(defn execute!
  "Generates and applies migrations relted to entities on database.
   All mimgrations run in a single transaction"
  [{:keys [datasource prev-sql-schema next-sql-schema migrations-table]}]
  (let [hsql-statements (hsql-statements prev-sql-schema next-sql-schema)
        sql-statements (->sql-statements hsql-statements)
        insert-query (hsql/format {:insert-into migrations-table
                                   :columns [:sql-schema :hsql-statements :sql-statements]
                                   :values [[[:lift next-sql-schema] [:lift hsql-statements] [:lift sql-statements]]]})]
    (try
      (when (seq sql-statements)
        (jdbc/with-transaction [tx datasource]
          (doseq [sql-statement (conj sql-statements insert-query)]
            (uc/tap->> "Running migration" sql-statement)
            (jdbc/execute! tx sql-statement))))
      (catch Throwable t
        (uc/tap->> "Apply migration error" t)
        (throw t)))))

(comment

  (def ds (jdbc/get-datasource {:jdbcUrl "jdbc:postgresql://localhost:6432/vadedb?user=vadeuser&password=vadepassword"}))

  (jdbc/execute-one! ds ["SELECT sql_schema FROM postgres.migrations_table LIMIT 1"]
    {:return-keys true
     :builder-fn jdbc.rs/as-unqualified-kebab-maps})

  (def prev-sql-schema {})

  (def next-sql-schema {})

  (def aenv {:datasource ds
             :connection (jdbc/get-connection ds)
             :nspace :pg
             :provider :postgres
             :migrations-table :pg.migrations-table})

  (execute! (assoc aenv
              :prev-sql-schema prev-sql-schema
              :next-sql-schema next-sql-schema))

  :rcf)
