(ns com.vadelabs.sql-core.provider
  (:require
   [com.vadelabs.sql-core.eql :as eql]
   [com.vadelabs.sql-core.spec :as spec]
   [com.vadelabs.utils-core.interface :as uc]
   [com.vadelabs.utils-str.interface :as ustr]
   [com.vadelabs.utils-exception.interface :as uex]
   [honey.sql :as hsql]
   [malli.core :as m]
   [next.jdbc.types :refer [as-other]]))

(defn ->hsql-value
  [schema k val]
  (try
    (m/schema schema)
    (catch Exception e
      (tap> {:e e
             :schema schema
             :k k :val val})))
  (let [enum-name (-> schema m/schema m/ast :keys k :properties :type-name)

        column-type (-> schema
                      m/schema
                      m/ast
                      :keys
                      k
                      :value
                      :type)]
    (cond
      (= column-type :enum) (as-other (name val))

      (uc/is-ident? val) (second val)

      (and (vector? val) (every? uc/is-ident? val)) [:lift (mapv second val)]

      (or (vector? val) (map? val)) [:lift val]

      (qualified-keyword? val) ((comp (partial ustr/join "/") (juxt namespace name)) val)

      (keyword? val) (name val)

      :else val)))

(def ^:private reserved
  #{:group :cardinality})

(defn safeguard
  [column-keyword]
  (if (contains? reserved column-keyword)
    [:quote column-keyword]
    column-keyword))

(defn ->column-keyword
  [k]
  (safeguard (-> k name keyword)))

(defn prepare-data
  [table-keyword data]
  (let [schema (spec/entity-schema table-keyword)]
    (reduce-kv (fn [{:keys [columns values] :as acc} k v]
                 (-> acc
                   (assoc :columns (conj columns (->column-keyword k)))
                   (assoc :values (conj values (->hsql-value schema k v)))))
      {:columns [] :values []}
      data)))
(defn ^:private referenced?
  [item]
  (and
    (vector? item) (or (uc/is-ident? item) (every? uc/is-ident? item))
    (seq item)))

(defn ->hsql-relations
  [nspace ident relations]
  (reduce-kv
    (fn [acc key ?idents]
      (let [table-keyword (uc/prefix-keyword "-" (namespace key) (name key))
            qualified-table-keyword (uc/prefix-keyword "." nspace table-keyword)
            values (cond
                     (uc/is-ident? ?idents) [[(second ident) (second ?idents)]]
                     (every? uc/is-ident? ?idents) (mapv
                                                     (fn [?ident] [(second ident) (second ?ident)])
                                                     ?idents))]
        (into acc (map (fn [v]
                         {:insert-into qualified-table-keyword
                          :columns [(-> key namespace keyword)  (-> key name keyword)]
                          :values v})
                    values))))
    []
    relations))

(defn unref-insert-statements
  [{:keys [nspace] :as aenv} delta]
  (->> delta
    (reduce-kv (fn [acc ident data]
                 (let [relations (uc/filter-vals referenced? data)
                       table-keyword (uc/prefix-keyword "." nspace (-> ident first namespace))
                       data (update-keys data (comp keyword name))
                       coerced-data (spec/coerce table-keyword data)
                       {:keys [columns values]} (prepare-data table-keyword coerced-data)
                       table-data {:insert-into table-keyword
                                   :columns columns
                                   :values values}
                       relation-data (->hsql-relations nspace ident relations)]
                   (conj acc table-data)))
      [])
    (uc/group-by (juxt :insert-into :columns))
    (reduce-kv (fn [acc [table-keyword columns] values]
                 (conj acc {:insert-into table-keyword
                            :columns columns
                            :on-conflict [] ;; TODO needs to fix it for unique cols
                            :do-nothing true ;; TODO update set if needed
                            :values (mapv :values values)}))
      [])
    (mapv hsql/format)))

(defn ref-insert-statements
  [{:keys [nspace] :as aenv} delta]
  (->> delta
    (reduce-kv (fn [acc ident data]
                 (let [relations (uc/filter-vals referenced? data)
                       table-keyword (uc/prefix-keyword "." nspace (-> ident first namespace))
                       data (update-keys data (comp keyword name))
                       coerced-data (spec/coerce table-keyword data)
                       {:keys [columns values]} (prepare-data table-keyword coerced-data)
                       table-data {:insert-into table-keyword
                                   :columns columns
                                   :values values}
                       relation-data (->hsql-relations nspace ident relations)]
                   (into acc relation-data)))
      [])
    (uc/group-by (juxt :insert-into :columns))
    (reduce-kv (fn [acc [table-keyword columns] values]
                 (conj acc {:insert-into table-keyword
                            :columns columns
                            :on-conflict [] ;; TODO needs to fix it for unique cols
                            :do-nothing true ;; TODO update set if needed
                            :values (mapv :values values)}))
      [])
    (mapv hsql/format)))

(defn delta->insert-statements
  [aenv delta]
  (uc/concat-vec
    (unref-insert-statements aenv delta)
    (ref-insert-statements aenv delta)))

(defn for-save
  [aenv {:keys [delta data]}]
  (let [delta (or delta (uc/prepare-for-save data))]
    (delta->insert-statements aenv delta)))

(defn for-query
  [aenv eql-query]
  (->> eql-query
    (eql/->hsql aenv)
    hsql/format))

(defn for-delete
  [aenv pparams])
