(ns com.vadelabs.sql-core.spec
  (:require
   [com.vadelabs.sql-core.types :as types]
   [com.vadelabs.utils-exception.interface :as uex]
   [malli.core :as m]
   [malli.error :as me]
   [malli.generator :as mg]
   [malli.registry :as mr]
   [malli.transform :as mt]
   [muotti.core :as muotti]
   [muotti.malli :as mm]))

(def registry*
  "Atom to hold malli schemas"
  (atom (merge
          (m/default-schemas)
          {:ref          types/reference
           :smallint     types/smallint
           :bigint       types/bigint
           :integer      types/integer
           :decimal      types/decimal
           :numeric      types/numeric
           :real         types/real
           :double       types/double
           :smallserial  types/smallserial
           :serial       types/serial
           :bigserial    types/bigserial
           :char         types/char
           :jsonb        types/jsonb
           :timestamp    types/timestamp
           :timestamp-tz types/timestamp-tz
           :date         types/date
           :time         types/time
           :time-tz      types/time-tz
           :interval     types/interval
           :array        types/array})))

(def registry
  "Mutable malli registry to hold all defined by user and utility schemas"
  (mr/mutable-registry registry*))

(defn register!
  "Adds entity and its reference schemas to the global registry"
  [schema-keyword ?ast]
  (let [spec (if (map? ?ast)
               (m/from-ast ?ast {:registry registry})
               ?ast)]
    (swap! registry* assoc schema-keyword spec)
    spec))

(defn valid?
  "Check if the data is aligned with the spec"
  [schema-keyword data]
  (or (m/validate schema-keyword data {:registry registry})
    (uex/raise :type uex/incorrect
      :code :invalid-spec
      :error (me/humanize (m/explain schema-keyword data {:registry registry}))
      :hint (str "Invalid data for schema " schema-keyword))))

(def ^:private malli-transformer
  (mm/transformer (muotti/->transformer mm/malli-config)))

(defn coerce
  "Coerce data fields according to schema"
  [schema-keyword data]
  (let [schema (mr/schema registry schema-keyword)]
    (if schema
      (try (m/decode schema data {:registry registry} (mt/transformer
                                                        malli-transformer
                                                        mt/strip-extra-keys-transformer))
        (catch Exception e
          (tap> {:e e :schema schema :data data})
          (throw e)))
      data)))

(defn entity-schema
  [schema-keyword]
  (let [schema (mr/schema registry schema-keyword)]
    schema))

(defn sample
  "Creates single random entity sample which should match a schema"
  [schema-keyword]
  (let [schema (mr/schema registry schema-keyword)]
    (mg/generate schema {:seed (rand-int 100) :registry registry})))

(defn samples
  "Creates n number of random entity samples which should match a schema"
  [schema-keyword n]
  (let [schema (mr/schema registry schema-keyword)]
    (mg/sample schema {:seed (rand-int 100) :size n :registry registry})))
