(ns com.vadelabs.datasource-xero.interface
  (:require
   [com.vadelabs.datasource-core.interface :as dc]
   [com.vadelabs.rest-core.interface :as rc]
   [com.vadelabs.utils-core.interface :as uc]
   [com.vadelabs.utils-str.interface :as ustr]))

(def apis
  [:accounting :app-store :assets :bankfeeds :files :finance
   :identity :payroll-au :payroll-nz :payroll-uk :projects])

(def definitions
  (->> apis
    (reduce #(assoc %1 %2 (uc/yaml-resource (ustr/format "datasource-xero/%s.yaml" (name %2)))) {})))

(def handlers
  (->> apis
    (reduce #(assoc %1 %2 (-> definitions %2 rc/openapi->handlers)) {})))

(def base-urls
  (->> apis
    (reduce #(assoc %1 %2 (-> definitions %2 rc/openapi->handlers)) {})))

(defn add-auth-header
  [{:keys [token xero-tenant-id]}]
  {:name :xero/add-auth-header
   :enter (fn [ctx]
            (update-in ctx [:request :headers]
              (fn [headers]
                (cond-> headers
                  token (assoc "Authorization" (str "Bearer " token))
                  xero-tenant-id (assoc "Xero-Tenant-Id" xero-tenant-id)))))})

(defmethod dc/http-clent ::xero
  [{:keys [provider config]}]
  (let [{:keys [base-url]} config
        datasource (-> provider uc/namify uc/keywordize)
        handlers (datasource handlers)
        base-url (or base-url (datasource base-urls))
        base-url (ustr/rtrim base-url "/")
        interceptors dc/perform-sync-request]
    (rc/bootstrap base-url handlers {:interceptors interceptors :use-defaults? true})))

(defmethod dc/oauth2-client ::xero
  [{:keys [http-client]} access-tokens]
  (let [{:keys [base-url handlers]} http-client
        add-auth-header (add-auth-header access-tokens)
        interceptors (rc/inject dc/perform-sync-request add-auth-header :before :dci/encode-request-body)]
    (rc/bootstrap base-url handlers {:interceptors interceptors :use-defaults? true})))

(defmethod dc/oauth2-config ::xero
  [{:keys [client-id client-secret host-root-url landing-uri]
    :or {host-root-url "http://localhost:3000"
         client-id "590081465679-qruvellphd6tuhqbgj08n5s6vnq0s4mv.apps.googleusercontent.com"
         client-secret "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"
         landing-uri "/"}}]
  {:provider :xero
   :authorize-uri    "https://accounts.google.com/o/oauth2/v2/auth"
   :access-token-uri "https://oauth2.googleapis.com/token"
   :client-id        client-id
   :client-secret    client-secret
   :scopes           ["https://www.googleapis.com/auth/userinfo.email"
                      "https://www.googleapis.com/auth/userinfo.profile"]
   :launch-uri       "/oauth2/google/login"
   :redirect-uri     (str host-root-url "/oauth2/google/callback")
   :landing-uri      landing-uri
   :label "Continue With Google"})

(defmethod dc/attributes ::xero
  [{:keys [provider]}]
  (->> provider
    name
    keyword
    (get handlers)
    dc/handlers->attributes))

(defmethod dc/actions ::xero
  [{:keys [provider]}]
  (->> provider
    name
    keyword
    (get handlers)
    dc/handlers->actions))

(derive :xero/accounting ::xero)
(derive :xero/app-store ::xero)
(derive :xero/assets ::xero)
(derive :xero/bankfeeds ::xero)
(derive :xero/files ::xero)
(derive :xero/finance ::xero)
(derive :xero/identity ::xero)
(derive :xero/payroll-au ::xero)
(derive :xero/payroll-nz ::xero)
(derive :xero/payroll-uk ::xero)
(derive :xero/projects ::xero)

(def adapters
  (reduce
    (fn [acc ds]
      (assoc acc ds
        {:adapter/id (-> :adapter.xero (uc/nspacify ds) uc/uuid)
         :adapter/type :adapter.type/rest
         :adapter/provider (uc/nspacify :xero ds)
         :adapter/nspace (->> ds uc/namify first
                           (ustr/str "x")
                           uc/keywordize)
         :adapter/qualify-attributes true
         :adapter/config {}}))
    {}
    apis))

(def datasources
  [{:datasource/id (-> :datasource.xero (uc/nspacify :accounting) uc/uuid)
    :datasource/display-name "Xero Accounting API"
    :datasource/description "The Accounting API exposes accounting and related functions of the main Xero application and can be used for a variety of purposes such as creating transactions like invoices and credit notes, right through to extracting accounting data via our reports endpoint."
    :datasource/slug "xero-accounting"
    :datasource/icon "https://upload.wikimedia.org/wikipedia/en/thumb/9/9f/Xero_software_logo.svg/440px-Xero_software_logo.svg.png"
    :datasource/preview "https://www.holliesbookkeeping.co.uk/media/2018/04/xero_beautifulaccountingsoftware-1000x380.jpg"
    :datasource/adapter (:accounting adapters)
    :datasource/attributes (dc/attributes {:provider :xero/accounting})
    :datasource/actions (dc/actions {:provider :xero/accounting})
    :datasource/categories []
    :datasource/collections []}
   {:datasource/id (-> :datasource.xero (uc/nspacify :app-store) uc/uuid)
    :datasource/display-name "Xero AppStore API"
    :datasource/description "These endpoints are for Xero Partners to interact with the App Store Billing platform"
    :datasource/slug "xero-app-store"
    :datasource/icon "https://upload.wikimedia.org/wikipedia/en/thumb/9/9f/Xero_software_logo.svg/440px-Xero_software_logo.svg.png"
    :datasource/preview "https://www.holliesbookkeeping.co.uk/media/2018/04/xero_beautifulapp-storesoftware-1000x380.jpg"
    :datasource/adapter (:app-store adapters)
    :datasource/attributes (dc/attributes {:provider :xero/app-store})
    :datasource/actions (dc/actions {:provider :xero/app-store})
    :datasource/categories []
    :datasource/collections []}
   {:datasource/id (-> :datasource.xero (uc/nspacify :assets) uc/uuid)
    :datasource/display-name "Xero Assets API"
    :datasource/description "The Assets API exposes fixed asset related functions of the Xero Accounting application and can be used for a variety of purposes such as creating assets, retrieving asset valuations etc."
    :datasource/slug "xero-assets"
    :datasource/icon "https://upload.wikimedia.org/wikipedia/en/thumb/9/9f/Xero_software_logo.svg/440px-Xero_software_logo.svg.png"
    :datasource/preview "https://www.holliesbookkeeping.co.uk/media/2018/04/xero_beautifulassetssoftware-1000x380.jpg"
    :datasource/adapter (:assets adapters)
    :datasource/attributes (dc/attributes {:provider :xero/assets})
    :datasource/actions (dc/actions {:provider :xero/assets})
    :datasource/categories []
    :datasource/collections []}
   {:datasource/id (-> :datasource.xero (uc/nspacify :bankfeeds) uc/uuid)
    :datasource/display-name "Xero Bank Feeds API"
    :datasource/description "The Bank Feeds API is a closed API that is only available to financial institutions that have an established financial services partnership with Xero. If you're an existing financial services partner that wants access, contact your local Partner Manager. If you're a financial institution who wants to provide bank feeds to your business customers, contact us to become a financial services partner."
    :datasource/slug "xero-bankfeeds"
    :datasource/icon "https://upload.wikimedia.org/wikipedia/en/thumb/9/9f/Xero_software_logo.svg/440px-Xero_software_logo.svg.png"
    :datasource/preview "https://www.holliesbookkeeping.co.uk/media/2018/04/xero_beautifulaccountingsoftware-1000x380.jpg"
    :datasource/adapter (:bankfeeds adapters)
    :datasource/attributes (dc/attributes {:provider :xero/bankfeeds})
    :datasource/actions (dc/actions {:provider :xero/bankfeeds})
    :datasource/categories []
    :datasource/collections []}
   {:datasource/id (-> :datasource.xero (uc/nspacify :files) uc/uuid)
    :datasource/display-name "Xero Files API"
    :datasource/description "These endpoints are specific to Xero Files API"
    :datasource/slug "xero-files"
    :datasource/icon "https://upload.wikimedia.org/wikipedia/en/thumb/9/9f/Xero_software_logo.svg/440px-Xero_software_logo.svg.png"
    :datasource/preview "https://www.holliesbookkeeping.co.uk/media/2018/04/xero_beautifulaccountingsoftware-1000x380.jpg"
    :datasource/adapter (:files adapters)
    :datasource/attributes (dc/attributes {:provider :xero/files})
    :datasource/actions (dc/actions {:provider :xero/files})
    :datasource/categories []
    :datasource/collections []}
   {:datasource/id (-> :datasource.xero (uc/nspacify :finance) uc/uuid)
    :datasource/display-name "Xero Finance API"
    :datasource/description "The Finance API is a collection of endpoints which customers can use in the course of a loan application, which may assist lenders to gain the confidence they need to provide capital"
    :datasource/slug "xero-finance"
    :datasource/icon "https://upload.wikimedia.org/wikipedia/en/thumb/9/9f/Xero_software_logo.svg/440px-Xero_software_logo.svg.png"
    :datasource/preview "https://www.holliesbookkeeping.co.uk/media/2018/04/xero_beautifulaccountingsoftware-1000x380.jpg"
    :datasource/adapter (:finance adapters)
    :datasource/attributes (dc/attributes {:provider :xero/finance})
    :datasource/actions (dc/actions {:provider :xero/finance})
    :datasource/categories []
    :datasource/collections []}
   {:datasource/id (-> :datasource.xero (uc/nspacify :identity) uc/uuid)
    :datasource/display-name "Xero OAuth 2 Identity Service API"
    :datasource/description "These endpoints are related to managing authentication tokens and identity for Xero API"
    :datasource/slug "xero-identity"
    :datasource/icon "https://upload.wikimedia.org/wikipedia/en/thumb/9/9f/Xero_software_logo.svg/440px-Xero_software_logo.svg.png"
    :datasource/preview "https://www.holliesbookkeeping.co.uk/media/2018/04/xero_beautifulaccountingsoftware-1000x380.jpg"
    :datasource/adapter (:identity adapters)
    :datasource/attributes (dc/attributes {:provider :xero/identity})
    :datasource/actions (dc/actions {:provider :xero/identity})
    :datasource/categories []
    :datasource/collections []}
   {:datasource/id (-> :datasource.xero (uc/nspacify :payroll-au) uc/uuid)
    :datasource/display-name "Xero Payroll AU API"
    :datasource/description "This is the Xero Payroll API for orgs in Australia region"
    :datasource/slug "xero-payroll-au"
    :datasource/icon "https://upload.wikimedia.org/wikipedia/en/thumb/9/9f/Xero_software_logo.svg/440px-Xero_software_logo.svg.png"
    :datasource/preview "https://www.holliesbookkeeping.co.uk/media/2018/04/xero_beautifulaccountingsoftware-1000x380.jpg"
    :datasource/adapter (:payroll-au adapters)
    :datasource/attributes (dc/attributes {:provider :xero/payroll-au})
    :datasource/actions (dc/actions {:provider :xero/payroll-au})
    :datasource/categories []
    :datasource/collections []}
   {:datasource/id (-> :datasource.xero (uc/nspacify :payroll-nz) uc/uuid)
    :datasource/display-name "Xero Payroll NZ"
    :datasource/description "This is the Xero Payroll API for orgs in the NZ region"
    :datasource/slug "xero-payroll-nz"
    :datasource/icon "https://upload.wikimedia.org/wikipedia/en/thumb/9/9f/Xero_software_logo.svg/440px-Xero_software_logo.svg.png"
    :datasource/preview "https://www.holliesbookkeeping.co.uk/media/2018/04/xero_beautifulaccountingsoftware-1000x380.jpg"
    :datasource/adapter (:payroll-nz adapters)
    :datasource/attributes (dc/attributes {:provider :xero/payroll-nz})
    :datasource/actions (dc/actions {:provider :xero/payroll-nz})
    :datasource/categories []
    :datasource/collections []}
   {:datasource/id (-> :datasource.xero (uc/nspacify :payroll-uk) uc/uuid)
    :datasource/display-name "Xero Payroll UK"
    :datasource/description "This is the Xero Payroll API for orgs in the UK region"
    :datasource/slug "xero-payroll-uk"
    :datasource/icon "https://upload.wikimedia.org/wikipedia/en/thumb/9/9f/Xero_software_logo.svg/440px-Xero_software_logo.svg.png"
    :datasource/preview "https://www.holliesbookkeeping.co.uk/media/2018/04/xero_beautifulaccountingsoftware-1000x380.jpg"
    :datasource/adapter (:payroll-uk adapters)
    :datasource/attributes (dc/attributes {:provider :xero/payroll-uk})
    :datasource/actions (dc/actions {:provider :xero/payroll-uk})
    :datasource/categories []
    :datasource/collections []}
   {:datasource/id (-> :datasource.xero (uc/nspacify :projects) uc/uuid)
    :datasource/display-name "Xero Projects API"
    :datasource/description "This is the Xero Projects API"
    :datasource/slug "xero-projects"
    :datasource/icon "https://upload.wikimedia.org/wikipedia/en/thumb/9/9f/Xero_software_logo.svg/440px-Xero_software_logo.svg.png"
    :datasource/preview "https://www.holliesbookkeeping.co.uk/media/2018/04/xero_beautifulaccountingsoftware-1000x380.jpg"
    :datasource/adapter (:projects adapters)
    :datasource/attributes (dc/attributes {:provider :xero/projects})
    :datasource/actions (dc/actions {:provider :xero/projects})
    :datasource/categories []
    :datasource/collections []}])

;; (def default-handlers
;;   [{:route-name :userinfo
;;     :summary "user info"
;;     :method :get
;;     :path-parts ["/api.xro/2.0/Organisation"]
;;     :produces ["application/json"]
;;     :consumes ["application/json"]}
;;    {:route-name :accounts
;;     :summary "list accounts"
;;     :method :get
;;     :path-parts ["/api.xro/2.0/Accounts"]
;;     :produces ["application/json"]
;;     :consumes ["application/json"]}
;;    {:route-name :tenants
;;     :summary "list tenants"
;;     :method :get
;;     :path-parts ["/connections"]
;;     :produces ["application/json"]
;;     :consumes ["application/json"]}
;;    {:route-name :invoices
;;     :summary "list invoices"
;;     :method :get
;;     :path-parts ["/api.xro/2.0/Invoices"]
;;     :produces ["application/json"]
;;     :consumes ["application/json"]}
;;    {:route-name :create-invoice
;;     :summary "create invoice"
;;     :method :post
;;     :path-parts ["/api.xro/2.0/Invoices"]
;;     :produces ["application/json"]
;;     :consumes ["application/json"]}
;;    {:route-name :contacts
;;     :summary "list contacts"
;;     :method :get
;;     :path-parts ["/api.xro/2.0/Contacts"]
;;     :produces ["application/json"]
;;     :consumes ["application/json"]}
;;    {:route-name :items
;;     :summary "list items"
;;     :method :get
;;     :path-parts ["/api.xro/2.0/Items"]
;;     :produces ["application/json"]
;;     :consumes ["application/json"]}])

;; (defn make-http-instance
;;   [{:keys [identifier config] :as params}]
;;   (let [{:keys [base-url]} config
;;         ;; fname (ustr/format "datasource-xero/%s.yaml" (uc/namify identifier))
;;         ;; definition (uc/yaml-resource fname)
;;         ;; handlers (rc/openapi->handlers definition)
;;         ;; base-url (or base-url (rc/openapi-base-url definition))
;;         base-url #_(ustr/rtrim base-url "/") "https://api.xero.com"
;;         interceptors dc/perform-sync-request]
;;     (dc/bootstrap base-url default-handlers
;;       {:interceptors interceptors :use-defaults? true})))

;; (defmethod dc/http-clent :xero
;;   [aenv]
;;   (make-http-instance aenv))

;; (defn get-tenant-id
;;   [token]
;;   (let [add-auth-header (add-auth-header {:token token})
;;         interceptors (dc/inject dc/perform-sync-request add-auth-header :before :dci/encode-request-body)
;;         http-client (dc/bootstrap "https://api.xero.com" default-handlers {:interceptors interceptors :use-defaults? true})]
;;     (-> http-client
;;       (dc/response-for :tenants {})
;;       :body
;;       first
;;       :tenantId)))

;; (defmethod dc/oauth2-client :xero
;;   [{:keys [http-instance identifier config]} {:keys [route-name access-tokens]}]
;;   (let [{:keys [base-url]} config
;;         fname (ustr/format "datasource-xero/%s.yaml" (uc/namify identifier))
;;         definition (uc/yaml-resource fname)
;;         handlers (rc/openapi->handlers definition)
;;         base-url (or base-url (rc/openapi-base-url definition))
;;         base-url #_(ustr/rtrim base-url "/") "https://api.xero.com"
;;         {:keys [token]} access-tokens
;;         xero-tenant-id (get-tenant-id token) #_"9f1f6d87-5051-45e6-9e26-b31ad869861c"
;;         add-auth-header (add-auth-header {:token token :xero-tenant-id xero-tenant-id})
;;         interceptors (dc/inject dc/perform-sync-request add-auth-header :before :dci/encode-request-body)]
;;     (try
;;       (println "TOKEN VAL" (get-tenant-id token))
;;       (catch Exception e
;;         (println "exception occured")))
;;     (dc/bootstrap base-url default-handlers
;;       {:interceptors interceptors :use-defaults? true})))

;; (defmethod dc/oauth2-config :xero
;;   [{:keys [client-id client-secret host-root-url landing-uri]
;;     :or {host-root-url "http://localhost:3000"
;;          client-id "652217A27D584682B6658F457B10C4CB"
;;          client-secret "A3qsIvyAkzwxjxZJRav-l8L8-BW0ZLoKZqYUIjitLk7YOSPt"
;;          landing-uri "/"}}]
;;   {:authorize-uri "https://login.xero.com/identity/connect/authorize"
;;    :access-token-uri "https://identity.xero.com/connect/token"
;;    :client-id client-id
;;    :client-secret client-secret
;;    :scopes ["email" "openid" "profile"]
;;    :launch-uri "/oauth2/xero/login"
;;    :redirect-uri (str host-root-url "/oauth2/xero/callback")
;;    :landing-uri landing-uri
;;    :label "Continue With Xero"})

;; (comment
;;   (def hinst (dc/http-clent {:identifier :xero/hello}))
;;   (dc/request-for hinst :create-invoice {:Invoices []})

;;   :rcf)
