(ns com.vadelabs.turbo-css.sheet
  (:require
   [com.vadelabs.utils-core.interface :as uc]
   [goog.dom :as dom]
   [goog.object :as gobj]))

(def dev?
  ^boolean goog.DEBUG)
(def ^:private limit
  65534)

(defn- make-style-tag []
  (let [tag (dom/createElement "style")
        head (aget (dom/getElementsByTagNameAndClass "head") 0)]
    (gobj/set tag "type" "text/css")
    (dom/appendChild tag (dom/createTextNode ""))
    (dom/appendChild head tag)
    tag))

(defprotocol ISheet
  (insert! [this css cls-name])
  (flush! [this])
  (filled? [this]))

(deftype Sheet [tag sheet cache]
  ISheet
  (insert! [this rule cls-name]
    (when (filled? this)
      (throw (js/Error. (str "A stylesheet can only have " limit " rules"))))
    (when-not (@cache cls-name)
      (swap! cache conj cls-name)
      (let [rule (if (ifn? rule) (rule) rule)
            rules-count (gobj/get (gobj/get sheet "cssRules") "length")]
        (if dev?
          (dom/appendChild tag (dom/createTextNode rule))
          (try
            (.insertRule sheet rule rules-count)
            (catch :default e
              (when dev?
                (js/console.warn "Illegal CSS rule" rule))))))))
  (flush! [this]
    (-> tag
      .-parentNode
      (.removeChild tag)))
  (filled? [this]
    (= (count @cache) limit)))

(defn create-sheet []
  (let [tag (make-style-tag)
        sheet (gobj/get tag "sheet")]
    (Sheet. tag sheet (atom #{}))))

(def ^:private sheets
  (atom (list (create-sheet))))

(defn remove-styles! []
  (run! flush! @sheets)
  (reset! sheets (list (create-sheet))))

(defn escape-val [rule val]
  (if (= rule :content)
    (pr-str val)
    val))

(defn build-css [cls styles]
  (->> styles
    (map (fn [[rule val]] (str (name rule) ":" (escape-val rule val) ";")))
    (uc/str-join "")
    (#(str "." cls "{" % "}"))))

(defn css
  "Takes class name, static styles and dynamic styles.
   Injects styles and returns a string of generated class names."
  [cls static vars]
  (let [static (if (string? static) [static] static)
        sheet (first @sheets)]
    (if (filled? sheet)
      (do
        (swap! sheets conj (create-sheet))
        (css cls static vars))
      (do
        (loop [[s & static] static
               idx 0]
          (let [cls (str cls "-" idx)]
            (insert! sheet s cls)
            (when-not (empty? static)
              (recur static (inc idx)))))
        (if (pos? (count vars))
          (let [var-cls (str "vars-" (hash vars))]
            (insert! sheet #(build-css var-cls vars) var-cls)
            (str cls " " var-cls))
          cls)))))
