(ns com.vadelabs.utils-core.interface
  (:refer-clojure :exclude [group-by uuid str uuid?])
  (:require
   [clojure.set :as cset]
   [clojure.walk :as cwalk]
   [com.vadelabs.utils-core.anomaly :as anomaly]
   [com.vadelabs.utils-core.data :as data]
   [com.vadelabs.utils-core.diff :as diff]
   [com.vadelabs.utils-core.exceptions :as ex]
   [com.vadelabs.utils-core.inflect :as inflect]
   [com.vadelabs.utils-core.json :as json]
   [com.vadelabs.utils-core.patch :as patch]
   [com.vadelabs.utils-core.spec :as spec]
   [com.vadelabs.utils-core.sql :as sql]
   [com.vadelabs.utils-core.string :as str]
   [com.vadelabs.utils-core.template :as template]
   [com.vadelabs.utils-core.transformer :as utransformer]
   [com.vadelabs.utils-core.transit :as transite]
   [com.vadelabs.utils-core.uri :as uri]
   [com.vadelabs.utils-core.uuid :as uuid]
   [com.vadelabs.utils-str.interface :as ustr]
   #?@(:clj [[com.vadelabs.utils-core.yaml :as yaml]
             [clojure.java.io :as io]])
   [fipp.edn :as fpp]
   [pyramid.core :as p])
  #?(:cljs (:require-macros [com.vadelabs.utils-core.interface])))

(defmacro ok->
  "When expr is not anomaly, threads it into the first form (via ->),
  and when that result is not an anomaly, through the next etc"
  [expr & forms]
  `(anomaly/ok-> ~expr ~@forms))

(defmacro ok->>
  "When expr is not an anomaly, threads it into the first form (via ->>),
  and when that result is not an anomaly, through the next etc"
  [expr & forms]
  `(anomaly/ok->> ~expr ~@forms))

(defmacro raise
  [& args]
  `(ex/raise ~@args))

(defn not-found
  [& args]
  (apply anomaly/not-found args))

(defn keywordize
  [& args]
  (apply data/keywordize args))

(defn symbolize
  [& args]
  (apply data/symbolize args))

(defn namify
  [& args]
  (apply data/namify args))

(defn stringify
  [s]
  (-> s
    ustr/str
    (ustr/ltrim ":")))

(defn nspacify
  ([kw] (namespace kw))
  ([nspace coll]
   (cwalk/prewalk
     #(if (keyword? %) (keywordize nspace %) %)
     coll)))

#?(:clj
   (defmethod print-method clojure.lang.PersistentQueue [q, w]
     ;; Overload the printer for queues so they look li2ke fish
     (print-method '<- w)
     (print-method (seq q) w)
     (print-method '-< w)))

(defn queue
  ([] #?(:clj clojure.lang.PersistentQueue/EMPTY :cljs #queue []))
  ([a] (into (queue) [a]))
  ([a & more] (into (queue) (cons a more))))

(defmacro str
  [& params]
  `(str/concat ~@params))

(defn str-format
  [& args]
  (apply str/format args))

(defn str-upper
  [& args]
  (apply str/upper args))

(defn str-blank?
  [& args]
  (apply str/blank? args))

(defn str-starts-with?
  [& args]
  (apply str/starts-with? args))
(defn str-split
  [& args]
  (apply str/split args))

(defn str-join
  [& args]
  (apply str/join args))

(defn str-index-of
  [& args]
  (apply str/index-of args))

(defn str-capital
  [& args]
  (apply str/capital args))

(defn str-camel
  [& args]
  (apply str/camel args))

(defn str-css-selector
  [& args]
  (apply str/css-selector args))

(defn str-ends-with?
  [& args]
  (apply str/ends-with? args))

(defn str-lower
  [& args]
  (apply str/lower args))
(defn str-replace
  [& args]
  (apply str/replace args))

(defn without-nils
  [& args]
  (apply data/without-nils args))

(defn str-trim
  [& args]
  (apply str/trim args))
(defn str-includes?
  [& args]
  (apply str/includes? args))

(defn str-split-lines
  [& args]
  (apply str/lines args))

(defn uuid
  [& args]
  (apply uuid/uuid args))

(defn uuid?
  [s]
  (uuid/uuid? s))

(defn uuid-next
  [& args]
  (apply uuid/next args))

(defn tempid
  [& args]
  (apply uuid/tempid args))

(defn tempid?
  [& args]
  (apply uuid/tempid? args))

(defn resolve-tempids
  [& args]
  (apply uuid/resolve-tempids args))

(defn spec-conform
  [& args]
  (apply spec/conform args))

(defn sql-join
  [& args]
  (apply sql/join args))

(defn sql-literal
  [& args]
  (apply sql/literal args))

(defn sql-name
  [& args]
  (apply sql/name args))

(defn transit-encode
  [& args]
  (apply transite/encode-str args))

(defn transit-decode
  [& args]
  (apply transite/decode-str args))

(defn json-encode
  [& args]
  (apply json/encode args))

(defn json-decode
  [& args]
  (apply json/decode args))

(defn concat-vec
  [& args]
  (apply data/concat-vec args))

(defn concat-set
  [& args]
  (apply data/concat-set args))

(defn diff
  "Returns a vector containing the differing, and non-existant elements, of
  two clojure datastructures."
  [state new-state]
  [(diff/alterations state new-state)
   (diff/removals state new-state)])

(defn patch
  "Applies a diff, as created by the diff function, to any datastructure."
  [state [alterations removals]]
  (-> state
    (patch/removals removals)
    (patch/alterations alterations)))

(defn changes
  "Returns all the keys that got changed"
  [prev-model next-model]
  (let [[alterations removals] (diff prev-model next-model)]
    (-> (set (keys alterations))
      (cset/union (set (keys removals))))))

(defn created
  "Returns keys those are created"
  [changes prev-model next-model]
  (filterv (fn [schema]
             (and (contains? next-model schema)
               (not (contains? prev-model schema))))
    changes))

(defn dropped
  "Return keys those are dropped in the model"
  [changes prev-model next-model]
  (filterv (fn [schema]
             (and (not (contains? next-model schema))
               (contains? prev-model schema)))
    changes))

(defn altered
  "Return keys those are altered in the model"
  [changes prev-model next-model]
  (filterv (fn [schema]
             (and (contains? next-model schema)
               (contains? prev-model schema)))
    changes))

(defn prefix-keyword
  [& args]
  (apply data/prefix-keyword args))

(defn prefix-str
  [& args]
  (apply data/prefix-str args))

(defn- assoc-some-transient! [m k v]
  (if (nil? v) m (assoc! m k v)))

(defn assoc-some
  "Associates a key k, with a value v in a map m, if and only if v is not nil."
  ([m k v]
   (if (nil? v) m (assoc m k v)))
  ([m k v & kvs]
   (loop [acc (assoc-some-transient! (transient (or m {})) k v)
          kvs kvs]
     (if (next kvs)
       (recur (assoc-some-transient! acc (first kvs) (second kvs)) (nnext kvs))
       (if (zero? (count acc))
         m
         (persistent! acc))))))

(defn plural
  [kw]
  (if (keyword? kw)
    (-> kw namify inflect/plural)
    (-> kw inflect/plural)))

(defn singular
  [kw]
  (if (keyword? kw)
    (-> kw namify inflect/singular)
    (-> kw inflect/singular)))

(defn coerce
  [& args]
  (apply utransformer/coerce args))

(defn ast-map-loc
  [& args]
  (apply data/ast-map-loc args))

(defn ast-map
  [& args]
  (apply data/mapz args))

(defn ast-filter
  [& args]
  (apply data/filterz args))

(defn group-by
  [& args]
  (apply data/group-by args))

(defn deep-merge
  [& args]
  (apply data/deep-merge args))

(defn ordered-map
  [& args]
  (apply data/ordered-map args))

(defn cva
  [{:keys [base variants defaults]}]
  (fn [{:keys [class] :as props}]
    (str-join " "
      (cond-> [base]
        variants (into (reduce-kv
                         (fn [acc variant-type variant-map]
                           (conj acc (get variant-map (get props variant-type (get defaults variant-type)))))
                         []
                         variants))
        (vector? class) (into class)
        (string? class) (conj class)))))

(defn uri
  [& args]
  (apply uri/uri args))

(defn urlify
  [& args]
  (->> args
    (map #(ustr/ltrim % "/"))
    (ustr/join "/")))

(defn template-tags
  [template]
  (template/tags template))

(defn template-render
  [& args]
  (apply template/render args))

#?(:clj
   (defmacro json-resource
     [resource-name]
     `(-> ~resource-name io/resource slurp json-decode)))

#?(:clj
   (defmacro yaml-resource
     [resource-name]
     `(-> ~resource-name io/resource slurp yaml/yaml->edn)))

#?(:clj
   (defn yaml->edn
     [& args]
     (apply yaml/yaml->edn args)))

(defn literal? [datum]
  (or
    (true? datum)
    (false? datum)
    (string? datum)
    (nil? datum)
    (symbol? datum)
    (number? datum)))

(defn is-ident?
  [x]
  (and (vector? x)
    (= 2 (count x))
    (qualified-keyword? (first x))
    (= "id" (name (first x)))
    (literal? (second x))))

#_(is-ident? (tempid [:person/id 1]))

(defn normalize
  [items]
  (let [items (if (map? items) [items] items)]
    (->> (p/db items)
      (mapcat (fn [[attr-identifier-key attr-by-identifier-key]]
                (for [[attr-identifier-val attr] attr-by-identifier-key]
                  [[attr-identifier-key attr-identifier-val] attr])))
      (into {})
      #_(cwalk/postwalk
          (fn [node]
            (if (is-ident? node)
              (tempid node)
              #_(let [suggestion (->> node
                                   (map stringify)
                                   (ustr/join "_"))]
                  [(first node) (uuid suggestion)])
              node))))))

(defn pprint-str
  [expr & {:keys [width level length]
           :or {width 110 level 8 length 25}}]
  (binding [*print-level* level
            *print-length* length]
    (with-out-str
      (fpp/pprint expr {:width width}))))

(defn pprint
  [expr & {:as opts}]
  (println (pprint-str expr opts)))

#_(normalize [{:workspace/id "1"
               :workspace/display-name "Your Workspace"}])

#_(normalize [{:profile/id "1"
               :profile/email "boom@shankar.com"
               :profile/identitities
               [{:identity/id "1"
                 :identity/provider :google}]
               :profile/workspaces
               [{:workspace/id "1"
                 :workspace/display-name "Your Workspace"}]}])
