(ns com.vadelabs.utils-core.data
  "Data manipulation and query helper functions"
  (:refer-clojure :exclude [read-string hash-map merge update-vals
                            parse-double group-by merge select-keys])
  (:require
   [clojure.zip :as cz]
   [com.vadelabs.utils-core.ds.ordered-map :as om]
   [com.vadelabs.utils-core.ds.ordered-set :as os]
   [com.vadelabs.utils-core.string :as ustr]
   #?@(:cljs [[cljs.core :as core]
              [cljs.reader :as r]]
       :clj [[clojure.core :as core]
             [clojure.edn :as r]])))

(defmacro fast-select-keys
  "A macro version of `select-keys`. Usefull when keys vector is known
  at compile time (aprox 600% performance boost).

  It is not 100% equivalent, this macro does not removes not existing
  keys in contrast to clojure.core/select-keys"
  [target keys]
  (assert (vector? keys) "keys expected to be a vector")
  `{~@(mapcat (fn [key] [key (list `core/get target key)]) keys) ~@[]})

(defmacro fast-get-in
  "A macro version of `get-in`. Usefull when the keys vector is known at
  compile time (20-40% performance improvement)."
  ([target keys]
   (assert (vector? keys) "keys expected to be a vector")
   `(-> ~target ~@(map (fn [key] (list `core/get key)) keys)))
  ([target keys default]
   (assert (vector? keys) "keys expected to be a vector")
   (let [last-index (dec (count keys))]
     `(-> ~target ~@(map-indexed (fn [index key]
                                   (if (= last-index index)
                                     (list `core/get key default)
                                     (list `core/get key)))
                      keys)))))

(defn ordered-map
  [& args]
  (apply om/init-ordered-map args))

(defn ordered-map?
  [item]
  (om/ordered-map? item))

(defn ordered-set
  [& args]
  (apply os/init-ordered-set args))

(defn ordered-set?
  [item]
  (os/ordered-set? item))

(defn editable-collection?
  [m]
  #?(:clj (instance? clojure.lang.IEditableCollection m)
     :cljs (implements? core/IEditableCollection m)))

(defn ^:private transient-concat
  [c1 colls]
  (loop [result (transient c1)
         colls  colls]
    (if colls
      (recur (reduce conj! result (first colls))
        (next colls))
      (persistent! result))))

(defn concat-set
  ([] #{})
  ([c1]
   (if (set? c1) c1 (into #{} c1)))
  ([c1 & more]
   (if (set? c1)
     (transient-concat c1 more)
     (transient-concat #{} (cons c1 more)))))

(defn concat-vec
  ([] [])
  ([c1]
   (if (vector? c1) c1 (into [] c1)))
  ([c1 & more]
   (if (vector? c1)
     (transient-concat c1 more)
     (transient-concat [] (cons c1 more)))))

(defn preconj
  [coll elem]
  (into [elem] coll))

(defn enumerate
  ([items] (enumerate items 0))
  ([items start]
   (loop [idx   start
          items items
          res   (transient [])]
     (if (empty? items)
       (persistent! res)
       (recur (inc idx)
         (rest items)
         (conj! res [idx (first items)]))))))

(defn seek
  ([pred coll]
   (seek pred coll nil))
  ([pred coll not-found]
   (reduce (fn [_ x]
             (if (pred x)
               (reduced x)
               not-found))
     not-found coll)))

(defn index-by
  "Return a indexed map of the collection keyed by the result of
  executing the getter over each element of the collection."
  ([kf coll] (index-by kf identity coll))
  ([kf vf coll]
   (persistent!
     (reduce #(assoc! %1 (kf %2) (vf %2)) (transient {}) coll))))

(defn index-of-pred
  [coll pred]
  (loop [c    (first coll)
         coll (rest coll)
         index 0]
    (if (nil? c)
      nil
      (if (pred c)
        index
        (recur (first coll)
          (rest coll)
          (inc index))))))

(defn index-of
  [coll v]
  (index-of-pred coll #(= % v)))

(defn replace-by-id
  ([value]
   (map (fn [item]
          (if (= (:id item) (:id value))
            value
            item))))
  ([coll value]
   (sequence (replace-by-id value) coll)))

(defn without-qualified
  [data]
  (into {} (remove (comp qualified-keyword? first)) data))

(defn without-keys
  "Return a map without the keys provided
  in the `keys` parameter."
  [data keys]
  (persistent!
    (reduce dissoc!
      (if (editable-collection? data)
        (transient data)
        (transient {}))
      keys)))

(defn remove-at-index
  "Takes a vector and returns a vector with an element in the
  specified index removed."
  [v index]
  ;; The subvec function returns a SubVector type that is an vector
  ;; but does not have transient impl, because of this, we need to
  ;; pass an explicit vector as first argument.
  (concat-vec []
    (subvec v 0 index)
    (subvec v (inc index))))

(defn zip [col1 col2]
  (map vector col1 col2))

(defn mapm
  "Map over the values of a map"
  ([mfn]
   (map (fn [[key val]] [key (mfn key val)])))
  ([mfn coll]
   (into {} (mapm mfn) coll)))

(defn deep-merge
  ([a b]
   (if (map? a)
     (merge-with deep-merge a b)
     b))
  ([a b & rest]
   (reduce deep-merge a (cons b rest))))

(defn update-vals
  "m f => {k (f v) ...}
  Given a map m and a function f of 1-argument, returns a new map where the keys of m
  are mapped to result of applying f to the corresponding values of m."
  [m f]
  (with-meta
    (persistent!
      (reduce-kv (fn [acc k v] (assoc! acc k (f v)))
        (if (editable-collection? m)
          (transient m)
          (transient {}))
        m))
    (meta m)))

(defn removev
  "Returns a vector of the items in coll for which (fn item) returns logical false"
  [fn coll]
  (filterv (comp not fn) coll))

(defn filterm
  "Filter values of a map that satisfy a predicate"
  [pred coll]
  (into {} (filter pred coll)))

(defn removem
  "Remove values of a map that satisfy a predicate"
  [pred coll]
  (into {} (remove pred coll)))

(defn map-perm
  "Maps a function to each pair of values that can be combined inside the
  function without repetition.

  Optional parameters:
  `pred?`   A predicate that if not satisfied won't process the pair
  `target?` A collection that will be used as seed to be stored

  Example:
  (map-perm vector [1 2 3 4]) => [[1 2] [1 3] [1 4] [2 3] [2 4] [3 4]]"
  ([mfn coll]
   (map-perm mfn (constantly true) [] coll))
  ([mfn pred? coll]
   (map-perm mfn pred? [] coll))
  ([mfn pred? target coll]
   (loop [result (transient target)
          current (first coll)
          coll (rest coll)]
     (if (not current)
       (persistent! result)
       (let [result
             (loop [result result
                    other (first coll)
                    coll (rest coll)]
               (if (not other)
                 result
                 (recur (cond-> result
                          (pred? current other)
                          (conj! (mfn current other)))
                   (first coll)
                   (rest coll))))]
         (recur result
           (first coll)
           (rest coll)))))))

(defn join-coll
  "Returns a new collection with the cartesian product of both collections.
  For example:
    (join [1 2 3] [:a :b]) => ([1 :a] [1 :b] [2 :a] [2 :b] [3 :a] [3 :b])
  You can pass a function to merge the items. By default is `vector`:
    (join [1 2 3] [1 10 100] *) => (1 10 100 2 20 200 3 30 300)"
  ([col1 col2] (join-coll col1 col2 vector []))
  ([col1 col2 join-fn] (join-coll col1 col2 join-fn []))
  ([col1 col2 join-fn acc]
   (cond
     (empty? col1) acc
     (empty? col2) acc
     :else (recur (rest col1) col2 join-fn
             (let [other (mapv (partial join-fn (first col1)) col2)]
               (concat acc other))))))

(def sentinel
  #?(:clj (Object.)
     :cljs (js/Object.)))

(defn getf
  "Returns a function to access a map"
  [coll]
  (partial get coll))

(defn update-in-when
  [m key-seq f & args]
  (let [found (get-in m key-seq sentinel)]
    (if-not (identical? sentinel found)
      (assoc-in m key-seq (apply f found args))
      m)))

(defn update-when
  [m key f & args]
  (let [found (get m key sentinel)]
    (if-not (identical? sentinel found)
      (assoc m key (apply f found args))
      m)))

(defn assoc-in-when
  [m key-seq v]
  (let [found (get-in m key-seq sentinel)]
    (if-not (identical? sentinel found)
      (assoc-in m key-seq v)
      m)))

(defn assoc-when
  [m key v]
  (let [found (get m key sentinel)]
    (if-not (identical? sentinel found)
      (assoc m key v)
      m)))

(defn domap
  "A side effect map version."
  ([f]
   (map (fn [x] (f x) x)))
  ([f coll]
   (map (fn [x] (f x) x) coll)))

(defn merge
  [& maps]
  (reduce conj (or (first maps) {}) (rest maps)))

(defn txt-merge
  "Text attrs specific merge function."
  [obj attrs]
  (reduce-kv (fn [obj k v]
               (if (nil? v)
                 (dissoc obj k)
                 (assoc obj k v)))
    obj
    attrs))

(defn distinct-xf
  [f]
  (fn [rf]
    (let [seen (volatile! #{})]
      (fn
        ([] (rf))
        ([result] (rf result))
        ([result input]
         (let [input* (f input)]
           (if (contains? @seen input*)
             result
             (do (vswap! seen conj input*)
               (rf result input)))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Data Parsing / Conversion
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn nan?
  [v]
  #?(:cljs (js/isNaN v)
     :clj  (not= v v)))

(defn- impl-parse-integer
  [v]
  #?(:cljs (js/parseInt v 10)
     :clj (try
            (Integer/parseInt v)
            (catch Throwable _
              nil))))

(defn- impl-parse-double
  [v]
  #?(:cljs (js/parseFloat v)
     :clj (try
            (Double/parseDouble v)
            (catch Throwable _
              nil))))

(defn parse-integer
  ([v]
   (parse-integer v nil))
  ([v default]
   (let [v (impl-parse-integer v)]
     (if (or (nil? v) (nan? v))
       default
       v))))

(defn parse-double
  ([v]
   (parse-double v nil))
  ([v default]
   (let [v (impl-parse-double v)]
     (if (or (nil? v) (nan? v))
       default
       v))))

(defn num-string? [v]
  ;; https://stackoverflow.com/questions/175739/built-in-way-in-javascript-to-check-if-a-string-is-a-valid-number
  #?(:cljs (and (string? v)
             (not (js/isNaN v))
             (not (js/isNaN (parse-double v))))

     :clj  (not= (parse-double v :nan) :nan)))

(defn read-string
  [v]
  (r/read-string v))

(defn coalesce-str
  [val default]
  (if (or (nil? val) (nan? val))
    default
    (str val)))

(defn coalesce
  [val default]
  (or val default))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Data Parsing / Conversion
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn nilf
  "Returns a new function that if you pass nil as any argument will
  return nil"
  [f]
  (fn [& args]
    (if (some nil? args)
      nil
      (apply f args))))

(defn nilv
  "Returns a default value if the given value is nil"
  [v default]
  (if (some? v) v default))

(defn any-key? [element & rest]
  (some #(contains? element %) rest))

(defn without-nils
  "Given a map, return a map removing key-value
  pairs when value is `nil`."
  [data]
  (into {} (remove (comp nil? second)) data))

(defn namify
  "Improved version of name that won't fail if the input is not a keyword"
  ([maybe-keyword] (namify maybe-keyword nil))
  ([maybe-keyword default-value]
   (cond
     (keyword? maybe-keyword)
     (name maybe-keyword)

     (string? maybe-keyword)
     maybe-keyword

     (nil? maybe-keyword) default-value

     :else
     (or default-value
       (str maybe-keyword)))))
#_(namespace "hello/hello")
#_(keyword "HELLO.workd/hello")
#_(keyword :hel)
#_(keyword? :hello/world)

(defn keywordize
  ([kw]
   (if (string? kw)
     (keyword kw)
     kw))
  ([nspace kw]
   (let [entity (cond
                  (string? kw) (-> kw keyword namespace)
                  (keyword? kw) (-> kw namespace))]
     (keywordize nspace entity kw)))
  ([nspace entity kw]
   (let [elems (cond-> []
                 nspace (conj nspace)
                 entity (conj entity))
         nspace (when (seq elems)
                  (->> elems
                    (map namify)
                    (ustr/join ".")))
         kw (namify kw)]
     (keyword nspace kw))))

#_(keywordize nil :world)

(defn symbolize
  ([kw]
   (cond
     (string? kw) (symbol kw)
     (keyword? kw) (symbol kw)
     :else kw))
  ([nspace kw]
   (let [entity (cond
                  (string? kw) (-> kw keyword namespace)
                  (keyword? kw) (-> kw namespace))]
     (symbolize nspace entity kw)))
  ([nspace entity kw]
   (let [elems (cond-> []
                 nspace (conj nspace)
                 entity (conj entity))
         nspace (->> elems
                  (map namify)
                  (ustr/join "."))
         kw (namify kw)]
     (symbol nspace kw))))

#_(symbolize :nspace :entity :kw)
#_(symbolize "nspace" "entity" "kw")
#_(symbolize "nspace" "kw")
#_(symbolize "nspace.entity" "kw")
#_(symbolize "nspace.entity/kw")
#_(symbolize :nspace.entity/kw)
#_(symbolize "nspace.entity/kw")

(defn with-next
  "Given a collection will return a new collection where each element
  is paired with the next item in the collection
  (with-next (range 5)) => [[0 1] [1 2] [2 3] [3 4] [4 nil]]"
  [coll]
  (map vector
    coll
    (concat (rest coll) [nil])))

(defn with-prev
  "Given a collection will return a new collection where each element
  is paired with the previous item in the collection
  (with-prev (range 5)) => [[0 nil] [1 0] [2 1] [3 2] [4 3]]"
  [coll]
  (map vector
    coll
    (concat [nil] coll)))

(defn with-prev-next
  "Given a collection will return a new collection where every item is paired
  with the previous and the next item of a collection
  (with-prev-next (range 5)) => [[0 nil 1] [1 0 2] [2 1 3] [3 2 4] [4 3 nil]]"
  [coll]
  (map vector
    coll
    (concat [nil] coll)
    (concat (rest coll) [nil])))

(defn prefix-str
  ([prefix kw]
   (prefix-str "-" prefix kw))
  ([sep prefix kw]
   (->> [prefix kw]
     (map namify)
     (ustr/join sep))))

(defn prefix-keyword
  "Given a keyword and a prefix will return a new keyword with the prefix attached
  (prefix-keyword \"prefix\" :test) => :prefix-test"
  ([& args]
   (->> args
     (apply prefix-str)
     keywordize)))

(defn ^:private extract-numeric-suffix
  [basename]
  (if-let [[_ p1 p2] (re-find #"(.*)-([0-9]+)$" basename)]
    [p1 (+ 1 (parse-integer p2))]
    [basename 1]))

(defn unique-name
  "A unique name generator"
  ([basename used]
   (unique-name basename used false))
  ([basename used prefix-first?]
   (assert (string? basename))
   (assert (set? used))

   (if (> (count basename) 1000)
     ;; We skip generating names for long strings. If the name is too long the regex can hang
     basename
     (let [[prefix initial] (extract-numeric-suffix basename)]
       (if (and (not prefix-first?)
             (not (contains? used basename)))
         basename
         (loop [counter initial]
           (let [candidate (if (and (= 1 counter) prefix-first?)
                             (str prefix)
                             (str prefix "-" counter))]
             (if (contains? used candidate)
               (recur (inc counter))
               candidate))))))))

(defn deep-mapm
  "Applies a map function to an associative map and recurses over its children
  when it's a vector or a map"
  [mfn m]
  (let [do-map
        (fn [entry]
          (let [[k v] (mfn entry)]
            (cond
              (or (vector? v) (map? v))
              [k (deep-mapm mfn v)]

              :else
              (mfn [k v]))))]
    (cond
      (map? m)
      (into {} (map do-map) m)

      (vector? m)
      (into [] (map (partial deep-mapm mfn)) m)

      :else
      m)))

(defn not-empty?
  [coll]
  (boolean (seq coll)))

(def boolean-or-nil?
  (some-fn nil? boolean?))

(defn kebab-keys [m]
  (->> m
    (deep-mapm
      (fn [[k v]]
        (if (or (keyword? k) (string? k))
          [(keyword (ustr/kebab (name k))) v]
          [k v])))))

(defn group-by
  ([kf coll] (group-by kf identity [] coll))
  ([kf vf coll] (group-by kf vf [] coll))
  ([kf vf iv coll]
   (let [conj (fnil conj iv)]
     (reduce (fn [result item]
               (update result (kf item) conj (vf item)))
       {}
       coll))))

(defn toggle-selection
  ([set value]
   (toggle-selection set value false))
  ([set value toggle?]
   (if-not toggle?
     (conj (ordered-set) value)
     (if (contains? set value)
       (disj set value)
       (conj set value)))))

(defn eq
  "Deep equality (works for JS objects in CLJS)"
  [o1 o2]
  #?(:clj (= o1 o2)
     :cljs
     (cond
       (not= (goog/typeOf o1)
         (goog/typeOf o2)) false
       ;; scalars
       (not (#{"array" "object"} (goog/typeOf o1))) (= o1 o2)
       ;; cljs data
       (= o1 o2) true
       ;; js array
       (js/Array.isArray o1) (and (= (count o1) (count o2))
                               (. o1 every #(eq %1 (aget o2 %2))))
       ;; js obj
       :else (let [ks1 (js/Object.keys o1)
                   ks2 (js/Object.keys o2)]
               (and
                 (= (count ks1) (count ks2))
                 (loop [ks ks1]
                   (cond
                     (empty? ks) true
                     (let [k (first ks)
                           v1 (unchecked-get o1 k)
                           v2 (unchecked-get o2 k)]
                       (eq v1 v2)) (recur (rest ks))
                     :else false)))))))

(defn ast-zipper
  "Make a zipper to navigate an AST tree of EDN query language"
  [ast]
  (->> ast
    (cz/zipper
      (fn branch? [x] (and (map? x)
                        (#{:root :join} (:type x))))
      (fn children [x] (:children x))
      (fn make-node [x xs] (assoc x :children (vec xs))))))

(defn ast-map-loc [f ast]
  (loop [loc (ast-zipper ast)]
    (if (cz/end? loc)
      (cz/root loc)
      (recur
        (cz/next
          (let [node (cz/node loc)]
            (if (= :root (:type node))
              loc
              (f loc))))))))

(defn filterz [f ast]
  (loop [loc (ast-zipper ast)]
    (if (cz/end? loc)
      (cz/root loc)
      (recur
        (cz/next
          (let [node (cz/node loc)]
            (if (f node)
              loc
              (cz/next (cz/remove loc)))))))))

(defn mapz [f ast]
  (loop [loc (ast-zipper ast)]
    (if (cz/end? loc)
      (cz/root loc)
      (recur
        (cz/next
          (let [node (cz/node loc)]
            (if (= :root (:type node))
              loc
              (cz/edit loc f))))))))
