(ns com.vadelabs.utils-core.ds.ordered-set
  (:require
   [com.vadelabs.utils-core.ds.ordered-map :refer [empty-ordered-map]]
   [com.vadelabs.utils-core.string :as ustr]
   #?(:cljs [cljs.reader :as reader]))
  #?(:clj
     (:import
      (clojure.lang Counted
        IObj
        IFn
        IHashEq
        IPersistentCollection
        IPersistentSet
        Reversible
        Seqable
        Sequential
        SeqIterator)
      (java.util Set)
      (java.lang Iterable))))

(declare empty-ordered-set)

(deftype OrderedSet [ordered-map]
  #?@(:clj
      [IPersistentSet
       (disjoin [_ k]
         (OrderedSet. (dissoc ordered-map k)))
       (contains [_ k]
         (contains? ordered-map k))
       (get [this k]
         (when (.contains this k) k))

       Set
       (size [this]
         (.count this))

       Iterable
       (iterator [this]
         (SeqIterator. (.seq this)))

       Comparable
       (compareTo [_ o]
         (compare (vec _) (vec o)))

       Counted

       IPersistentCollection
       (count [_]
         (count ordered-map))
       (cons [this o]
         (if (contains? ordered-map o)
           this
           (OrderedSet. (assoc ordered-map o nil))))
       (empty [_]
         empty-ordered-set)
       (equiv [this other]
         (or (identical? this other)
           (and (instance? Set other)
             (let [^Set s other]
               (and (= (.size this) (.size s))
                 (every? #(.contains s %) (.seq this)))))))
       Sequential
       Seqable
       (seq [_]
         (when-let [s (seq ordered-map)] (map key s)))

       Reversible
       (rseq [_]
         (when-let [s (rseq ordered-map)] (map key s)))

       IFn
       (invoke [this k]
         (get this k))

       IObj
       (meta [this]
         (.meta ^IObj ordered-map))
       (withMeta [this m]
         (OrderedSet. (.withMeta ^IObj ordered-map m)))

       IHashEq
       (hasheq [this] (.hasheq ^IHashEq (into #{} this)))

       Object
       (toString [this]
         (str "[" (ustr/join " " (map str this)) "]"))
       (hashCode [this]
         (.hashCode ^Object (into #{} this)))
       (equals [this other]
         (.equiv this other))]
      :cljs
      [Object
       (toString [this]
         (str "[" (ustr/join " " (map str this)) "]"))
       (equiv [this other]
         (-equiv this other))

       ICloneable
       (-clone [_] (OrderedSet. ordered-map))

       IWithMeta
       (-with-meta [coll meta] (OrderedSet. (with-meta ordered-map meta)))

       IMeta
       (-meta [coll] (meta ordered-map))

       ICollection
       (-conj [coll o]
         (OrderedSet. (assoc ordered-map o nil)))

       IEmptyableCollection
       (-empty [coll] (with-meta empty-ordered-set meta))

       IEquiv
       (-equiv [coll other]
         (and
           (set? other)
           (== (count coll) (count other))
           (every? #(contains? coll %)
             other)))

       IHash
       (-hash [coll] (hash (into #{} coll)))

       ISequential
       ISeqable
       (-seq [coll] (when-let [s (seq ordered-map)] (map key s)))

       IReversible
       (-rseq [coll] (when-let [s (rseq ordered-map)] (map key s)))

       ICounted
       (-count [coll] (-count ordered-map))

       IComparable
       (-compare [_ o]
         (compare (vec _) (vec o)))

       ILookup
       (-lookup [coll v]
         (-lookup coll v nil))
       (-lookup [coll v not-found]
         (if (-contains-key? ordered-map v)
           v
           not-found))

       ISet
       (-disjoin [coll v]
         (OrderedSet. (-dissoc ordered-map v)))

       IFn
       (-invoke [coll k]
         (-lookup coll k))
       (-invoke [coll k not-found]
         (-lookup coll k not-found))

       ;; IEditableCollection

       IPrintWithWriter
       (-pr-writer [coll writer opts]
         (-write writer (str "#ds/ordered-set " (into [] coll))))]))

#?(:clj
   (defmethod print-method OrderedSet [o ^java.io.Writer w]
     (.write w "#ds/ordered-set ")
     (print-method (into [] o) w)))

(def ^{:tag OrderedSet} empty-ordered-set
  (OrderedSet. empty-ordered-map))

(defn init-ordered-set
  ([] empty-ordered-set)
  ([& keys]
   (apply conj empty-ordered-set keys)))

(defn ordered-set?
  [os]
  (instance? OrderedSet os))

(def ->ordered-set
  (partial into empty-ordered-set))

#?(:cljs (reader/register-tag-parser! 'ds/ordered-set ->ordered-set))
