package com.whimsy.map.base;

import static org.fest.assertions.api.Assertions.assertThat;

import com.whimsy.map.api.Edge;
import com.whimsy.map.api.GeoPoint;

import edu.princeton.cs.algs4.Point2D;

/**
 * Created by whimsy on 6/3/15.
 */
public class Utils {


    public static double sqr(double x) {
        return x * x;
    }

    public static double distM(double lat1, double lon1, double lat2, double lon2) {
        return Math.sqrt(sqr(lat1 - lat2) + sqr(lon1 - lon2)) * Constant.GEO_SCALE;
    }


    public static double dist(double lat1, double lon1, double lat2, double lon2) {
        return Math.sqrt(sqr(lat1 - lat2) + sqr(lon1 - lon2));
    }

    /**
     * 计算坐标到边的最小距离距离
     * 1. 当有垂直映射点时， 以垂直距离为距离
     * 2. 当没有垂直映射点时， 取端点的最小值
     * @since 1.0.0
     *
     * @param lat  纬度
     * @param lon  经度
     * @param edge  道路轨迹边
     * @return
     */

    public static double distM(double lat, double lon, Edge edge) {
        double minDist = Constant.INF;


        // 遍历点距离
        for (Edge.Figure figure : edge.figures) {
            double tmpDist = distM(lat, lon, figure.lat, figure.lon);

            if (tmpDist < minDist) {
                minDist = tmpDist;
            }
        }

        // 遍历投影距离
        for (int i = 0; i < edge.figures.size() - 1; ++i) {
            Edge.Figure from = edge.figures.get(i);
            Edge.Figure to = edge.figures.get(i + 1);

            // 有投影
            if (cosAng(lat, lon, from.lat, from.lon, to.lat, to.lon) <= 0
                    && cosAng(lat, lon, to.lat, to.lon, from.lat, from.lon) <= 0) {

                double v1x = to.lat - from.lat;
                double v1y = to.lon - from.lon;
                double v2x = lat - from.lat;
                double v2y = lon - from.lon;

                double dist = Math.abs((v1x * v2y - v2x * v1y) / Math.sqrt(v1x * v1x + v1y * v1y));

                dist *= GeoPoint.geoScale;

                if (dist < minDist) {
                    minDist = dist;
                }
            }
        }

        return minDist;
    }

    /**
     * 计算 cos 值
     *
     *
     *  A, B, C 其中 ABC的外角的cos值。 180 - 角ABC
     *  Example :
     *     cos(Ax, Ay, Bx, By, Cx, Cy) <=0 && cos(Ax, By, Cx, Cy, Bx, By) <= 0 表明 A 在BC线段中有投影.
     * @param x0 A的x坐标
     * @param y0 A的y坐标
     * @param x1 B的x坐标
     * @param y1 B的y坐标
     * @param x2 C的x坐标
     * @param y2 C的y坐标
     * @return
     */
    public static double cosAng(double x0, double y0, double x1, double y1, double x2, double y2) {
        double v1x = x1 - x0;
        double v1y = y1 - y0;
        double v2x = x2 - x1;
        double v2y = y2 - y1;

        return (v1x * v2x + v1y * v2y) / Math.sqrt((v1x * v1x + v1y * v1y) * (v2x * v2x + v2y * v2y));
    }

    /**
     * 计算 线AB 和线 CD的交点
     *
     * @param A
     * @param B
     * @param C
     * @param D
     * @return  交点坐标
     */

    public static Point2D intersectPoint(Point2D A, Point2D B, Point2D C, Point2D D) {
        double s_abc = Point2D.area2(B, C, A);
        double s_abd = Point2D.area2(B, D, A);

        double s_cda = Point2D.area2(D, A, C);
        double s_cdb = Point2D.area2(D, B, C);


        double x = (s_cda) / (s_cda - s_cdb) * B.x() + (- s_cdb) / (s_cda - s_cdb) * A.x();
        double y = (s_abc) / (s_abc - s_abd) * D.y() + (- s_abd) / (s_abc - s_abd) * C.y();

        return new Point2D(x, y);
    }


    // TODO: to be tested.

    public static Point2D project(GeoPoint trajPoint, Edge edge) {
        return project(trajPoint, edge, new ValueBox<Double>(), new ValueBox<Double>());
    }


    // TODO : to be tested
    public static Point2D project(GeoPoint trajPoint, Edge edge,
                                  ValueBox<Double> distVertical, ValueBox<Double> distLeft) {
        double minDist = Constant.INF;


        double lat = trajPoint.lat;
        double lon = trajPoint.lon;

        Point2D res = null;

        double distSum = 0;


        Edge.Figure formerFigure = null;


        // 遍历点距离
        for (Edge.Figure figure : edge.figures) {
            double tmpDist = distM(lat, lon, figure.lat, figure.lon);


            if (formerFigure != null) {
                distSum += distM(formerFigure.lat, formerFigure.lon, figure.lat, figure.lon);
            }

            if (tmpDist < minDist) {
                minDist = tmpDist;
                res = new Point2D(figure.lat, figure.lon);
                distLeft.value = distSum;
            }

            formerFigure = figure;
        }


        distSum = 0;

        // 遍历投影距离
        for (int i = 0; i < edge.figures.size() - 1; ++i) {
            Edge.Figure from = edge.figures.get(i);
            Edge.Figure to = edge.figures.get(i + 1);


            // 有投影
            if (cosAng(lat, lon, from.lat, from.lon, to.lat, to.lon) <= 0
                    && cosAng(lat, lon, to.lat, to.lon, from.lat, from.lon) <= 0) {

                double v1x = to.lat - from.lat;
                double v1y = to.lon - from.lon;
                double v2x = lat - from.lat;
                double v2y = lon - from.lon;

                double dist = Math.abs((v1x * v2y - v2x * v1y) / Math.sqrt(v1x * v1x + v1y * v1y));

                dist *= GeoPoint.geoScale;

                if (dist < minDist) {
                    minDist = dist;

                    res = intersectPoint(new Point2D(from.lat, from.lon),
                                            new Point2D(to.lat, to.lon),
                                            new Point2D(lat, lon),
                                            new Point2D(lat + v1y, lon - v1x));

                    distLeft.value = distSum + distM(from.lat, from.lon, res.x(), res.y());
                }
            }

            distSum += distM(from.lat, from.lon, to.lat, to.lon);
        }

        distVertical.value = minDist;

        return res;
    }

    /**
     * 浮点数相等assertion
     * @param actual
     * @param expected
     */
    public static void isDoubleEqual(double actual, double expected) {
        assertThat(Math.abs(actual - expected) < Constant.EPS).isTrue();
    }

    /**
     * 球面距离相等assertion
     * @param actual
     * @param expected
     */
    public static void isGeoDistEqual(double actual, double expected) {
        assertThat(Math.abs(actual - expected) < 1).isTrue();
    }

}
