package com.whimsy.map.dataprocess;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.Reader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;

import edu.princeton.cs.algs4.Stopwatch;

/**
 * 从OSM下载的原始数据中, 提取出nodeOSM.txt & edgeOSM.txt
 *
 * 1.  提取出来的数据主要的不同点在于， 新提取出来的点， 不包括度数为2的点， 从而大大减少提取出的节点数目。
 *
 * 考虑效率没有使用DOM来解析XML
 * 采用JAXP中的StAX， 流式效率比较高
 *
 * Created by whimsy on 5/26/15.
 */
public class OSMProcess {

    Logger LOG = LoggerFactory.getLogger(OSMProcess.class);

    class Node {
        double lon;
        double lat;
    }

    class Way {
        public List<Long> enode = new ArrayList<Long>();
        public boolean isoneway;
    }

    Map<Long, Node> nodeMap = new HashMap<Long, Node>();
    List<Way> ways = new ArrayList<Way>();



    String nodeFile;   // absolute path
    String edgeFile;   // absolute path
    String sourceFile = "mapOSM.xml";   // relative path, OSM Raw XML data



    ListMultimap<Long, Integer> node2way = ArrayListMultimap.create();

    Set<Long> node2Extract = new HashSet<Long>();



    public OSMProcess() {


        String outputPath = this.getClass().getClassLoader().getResource("").getPath();

        nodeFile = outputPath + "/nodeOSM.txt";
        edgeFile = outputPath + "/edgeOSM.txt";
    }

    void parseFromXML() throws Exception{
        XMLInputFactory factory = XMLInputFactory.newFactory();

        LOG.debug(factory.toString());

        InputStream is = getClass().getClassLoader().getResourceAsStream(sourceFile);

        Reader reader = new InputStreamReader(is, "UTF-8");

        XMLEventReader r = factory.createXMLEventReader(reader);

        while (r.hasNext()) {
            XMLEvent e = r.nextEvent();
            //            LOG.info(e.toString());

            if (e.isStartElement()) {
                StartElement startElement = e.asStartElement();


                if (startElement.getName().getLocalPart().equals("node")) {
                    parseNode(startElement, r);
                }

                if (startElement.getName().getLocalPart().equals("way")) {
                    parseWay(startElement, r);
                }
            }

        }
    }

    void spliceAndExtract() throws Exception {
        for (int idx = 0; idx < ways.size(); ++idx) {
            Way way = ways.get(idx);

            // 头尾 处理
            if (nodeMap.get(way.enode.get(0)) == null) {
                throw new RuntimeException("Node ID that doesn't exist!");
            } else {
                node2Extract.add(way.enode.get(0));
            }

            if (nodeMap.get(way.enode.get(way.enode.size() - 1)) == null) {
                throw new RuntimeException("Node ID that doesn't exist!");
            } else {
                node2Extract.add(way.enode.get(way.enode.size() - 1));
            }


            for (int i = 1; i < way.enode.size() - 1; ++i) {
                if (nodeMap.get(way.enode.get(i)) == null) {
                    throw new RuntimeException("Node ID that doesn't exist!");
                } else {
                    node2way.put(way.enode.get(i), idx);
                }
            }
        }

        int innerPointCnt = 0;

        for (Long key : node2way.keySet()) {
            if (node2way.get(key).size() >= 2) {
                node2Extract.add(key);
            } else {
                ++innerPointCnt;
            }

        }

        LOG.info("Inner Point Cnt = {}", innerPointCnt);


        Map<Long, Integer> newIdDict = new HashMap<Long, Integer>();

        PrintWriter nodeOutput = new PrintWriter(nodeFile);

        int idCnt = 0;

        for (Long id : node2Extract) {
            Node node = nodeMap.get(id);
            nodeOutput.printf("%d\t%.7f\t%.7f\n", idCnt, node.lat, node.lon);
            newIdDict.put(id, idCnt++);
        }

        LOG.info("Extract Node number = {}", idCnt);
        nodeOutput.close();

        PrintWriter edgeOutput = new PrintWriter(edgeFile);

        idCnt = 0;

        for (Way way : ways) {
            int sid = 0;

            for (int j = 1; j < way.enode.size(); ++j) {
                Long idx = way.enode.get(j);

                if (node2Extract.contains(idx)) {
                    long start = way.enode.get(sid);
                    long end = idx;

                    int num = j - sid + 1;

                    edgeOutput.printf("%d\t%d\t%d\t%d", idCnt++, newIdDict.get(start), newIdDict.get(end), num);

                    for (int k = sid; k <= j; ++k) {
                        Node node = nodeMap.get(way.enode.get(k));

                        edgeOutput.printf("\t%.7f\t%.7f", node.lat, node.lon);
                    }

                    edgeOutput.println();


                    if (!way.isoneway) {
                        edgeOutput.printf("%d\t%d\t%d\t%d", idCnt++, newIdDict.get(end), newIdDict.get(start), num);

                        for (int k = j; k >= sid; --k) {
                            Node node = nodeMap.get(way.enode.get(k));

                            edgeOutput.printf("\t%.7f\t%.7f", node.lat, node.lon);
                        }
                        edgeOutput.println();
                    }

                    sid = j;

                }
            }

        }

        edgeOutput.close();

        LOG.info("Extract Edge number = {} (consider oneway)", idCnt);


    }

    void work() throws Exception {

        Stopwatch stopwatch = new Stopwatch();
        parseFromXML();
        LOG.info("ParseFromXML time elapsed {} sec", stopwatch.elapsedTime());

        LOG.info("The raw node number = {}", nodeMap.size());
        LOG.info("The raw way number = {} (not consider oneway..)", ways.size());


        stopwatch = new Stopwatch();
        spliceAndExtract();
        LOG.info("SpliceAndExtract time elapsed {} sec", stopwatch.elapsedTime());

    }

    private void parseWay(StartElement startElement, XMLEventReader r) throws Exception {


        int id = Integer.parseInt(startElement.getAttributeByName(new QName("id")).getValue());


        boolean isHighWay = false;
        boolean isOneWay  = false;

        List<Long> figure = new ArrayList<Long>();

        while (r.hasNext()) {
            XMLEvent e = r.nextEvent();
            if (e.isEndElement() && e.asEndElement().getName().getLocalPart().equals("way")) {
                break;
            }

            if (e.isStartElement()) {
                StartElement se = e.asStartElement();

                if (se.getName().getLocalPart().equals("nd")) {
                    Long nId = Long.parseLong(se.getAttributeByName(new QName("ref")).getValue());
                    figure.add(nId);

                }

                if (se.getName().getLocalPart().equals("tag")) {
                    if (se.getAttributeByName(new QName("k")).getValue().equals("highway")) {
                        isHighWay = true;
                    }

                    if (se.getAttributeByName(new QName("k")).getValue().equals("oneway")) {
                        isOneWay = true;
                    }
                }
            }

        }

        if (!isHighWay) {
            return;
        }

        Way way = new Way();

        way.enode = figure;
        way.isoneway = isOneWay;

        ways.add(way);

    }

    private void parseNode(StartElement startElement, XMLEventReader r) {
        Node node = new Node();

        Long id = Long.parseLong(startElement.getAttributeByName(new QName("id")).getValue());

        double lat = Double.parseDouble(startElement.getAttributeByName(new QName("lat")).getValue());
        double lon = Double.parseDouble(startElement.getAttributeByName(new QName("lon")).getValue());

        node.lat = lat;
        node.lon = lon;

        nodeMap.put(id, node);
    }

    public static void main(String [] args) throws Exception {

        new OSMProcess().work();

    }
}
