// Copyright 2015-2019 Workiva Inc.
// 
// Licensed under the Eclipse Public License 1.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// 
//      http://opensource.org/licenses/eclipse-1.0.php
// 
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package eva.error.v1;

import java.util.HashMap;
import java.util.Map;
import clojure.lang.Keyword;

public enum EvaErrorCode implements IErrorCode {

    DATASTRUCTURE_FAILURE(null,
                          Long.MIN_VALUE,
                          500,
                          "DatastructureFailure",
                          Keyword.intern("eva.error",
                                         "datastructure-failure"),
                          "null"),
        DATASTRUCTURE_SERIALIZATION_FAILURE(DATASTRUCTURE_FAILURE,
                                            4000,
                                            500,
                                            "DatastructureSerializationFailure",
                                            Keyword.intern("datastructure-failure",
                                                           "serialization-error"),
                                            "An error occurred while attempting to serialize an internal datastructure for storage."),
        DATASTRUCTURE_DESERIALIZATION_FAILURE(DATASTRUCTURE_FAILURE,
                                              4001,
                                              500,
                                              "DatastructureDeserializationFailure",
                                              Keyword.intern("datastructure-failure",
                                                             "deserialization-error"),
                                              "An error occurred while attempting to deserialize an internal datastructure from storage."),
        DATASTRUCTURE_CORRUPTION(DATASTRUCTURE_FAILURE,
                                 4100,
                                 500,
                                 "DatastructureCorruption",
                                 Keyword.intern("datastructure-failure",
                                                "corruption-suspected"),
                                 "Possible corruption detected in an internal datastructure."),
    UNKNOWN_ERROR(null,
                  -1,
                  500,
                  "UnknownError",
                  Keyword.intern("unknown-error",
                                 "unknown-error"),
                  "An uncategorized or unknown error occurred."),
    TIMEOUT_ERROR(null,
                  Long.MIN_VALUE,
                  500,
                  "TimeoutError",
                  Keyword.intern("eva.error",
                                 "timeout"),
                  "null"),
        STORAGE_TIMEOUT(TIMEOUT_ERROR,
                        6001,
                        504,
                        "StorageTimeout",
                        Keyword.intern("timeout",
                                       "backing-store"),
                        "Requests for data from the backing store timed out."),
        TRANSACTION_TIMEOUT(TIMEOUT_ERROR,
                            6002,
                            504,
                            "TransactionTimeout",
                            Keyword.intern("timeout",
                                           "transaction"),
                            "A transaction timed out."),
    API_ERROR(null,
              Long.MIN_VALUE,
              500,
              "APIError",
              Keyword.intern("eva.error",
                             "api-error"),
              "null"),
        INCORRECT_SYNTAX(API_ERROR,
                         Long.MIN_VALUE,
                         500,
                         "IncorrectSyntax",
                         Keyword.intern("api-error",
                                        "incorrect-syntax"),
                         "null"),
            INCORRECT_TRANSACT_SYNTAX(INCORRECT_SYNTAX,
                                      3000,
                                      400,
                                      "IncorrectTransactSyntax",
                                      Keyword.intern("api-error",
                                                     "incorrect-transact-syntax"),
                                      "Malformed transact request."),
            INCORRECT_QUERY_SYNTAX(INCORRECT_SYNTAX,
                                   3100,
                                   500,
                                   "IncorrectQuerySyntax",
                                   Keyword.intern("api-error",
                                                  "incorrect-query-syntax"),
                                   "Malformed query or rules."),
            INCORRECT_PULL_SYNTAX(INCORRECT_SYNTAX,
                                  3200,
                                  500,
                                  "IncorrectPullSyntax",
                                  Keyword.intern("api-error",
                                                 "incorrect-pull-syntax"),
                                  "Malformed pull pattern."),
            INCORRECT_CONNECT_SYNTAX(INCORRECT_SYNTAX,
                                     3300,
                                     500,
                                     "IncorrectConnectSyntax",
                                     Keyword.intern("api-error",
                                                    "incorrect-connect-syntax"),
                                     "Malformed connect input."),
            ILLEGAL_TEMP_ID(INCORRECT_SYNTAX,
                            3500,
                            500,
                            "IllegalTempID",
                            Keyword.intern("api-error",
                                           "illegal-argument"),
                            "Generic illegal argument syntax error."),
        COERCION_FAILURE(API_ERROR,
                         3900,
                         500,
                         "CoercionFailure",
                         Keyword.intern("api-error",
                                        "coercion-failure"),
                         "Failed to coerce value"),
        METHOD_NOT_YET_IMPLEMENTED(API_ERROR,
                                   3999,
                                   501,
                                   "MethodNotYetImplemented",
                                   Keyword.intern("api-error",
                                                  "not-yet-implemented"),
                                   "This method is not yet implemented."),
    INTERNAL_COMPONENT_FAILURE(null,
                               10000,
                               500,
                               "InternalComponentFailure",
                               Keyword.intern("eva",
                                              "internal-component-failure"),
                               "Internal component failure."),
    STORAGE_ERROR(null,
                  Long.MIN_VALUE,
                  500,
                  "StorageError",
                  Keyword.intern("eva.error",
                                 "storage-error"),
                  "null"),
        STORAGE_IO_FAILURE(STORAGE_ERROR,
                           Long.MIN_VALUE,
                           500,
                           "StorageIOFailure",
                           Keyword.intern("storage-error",
                                          "io-failure"),
                           "null"),
            INCOMPLETE_STORAGE_RESPONSE(STORAGE_IO_FAILURE,
                                        1,
                                        502,
                                        "IncompleteStorageResponse",
                                        Keyword.intern("storage-error",
                                                       "incomplete-response"),
                                        "The backing store is returning incomplete data."),
            STORAGE_NOT_CONNECTED(STORAGE_IO_FAILURE,
                                  2,
                                  503,
                                  "StorageNotConnected",
                                  Keyword.intern("storage-error",
                                                 "not-connected"),
                                  "Attempted to interact with storage without a connection"),
        STORAGE_STATE_MISMATCH(STORAGE_ERROR,
                               Long.MIN_VALUE,
                               500,
                               "StorageStateMismatch",
                               Keyword.intern("storage-error",
                                              "state-mismatch"),
                               "null"),
            DATA_NOT_FOUND(STORAGE_STATE_MISMATCH,
                           200,
                           404,
                           "DataNotFound",
                           Keyword.intern("storage-error",
                                          "data-not-found"),
                           "The storage backend indicates no such data exists."),
            STALE_LOCAL_DATA(STORAGE_STATE_MISMATCH,
                             201,
                             409,
                             "StaleLocalData",
                             Keyword.intern("storage-error",
                                            "stale-local-data"),
                             "Attempting to overwrite data based on stale information."),
            UNEXPECTED_STORAGE_REPLY(STORAGE_STATE_MISMATCH,
                                     202,
                                     500,
                                     "UnexpectedStorageReply",
                                     Keyword.intern("storage-error",
                                                    "unexpected-cas-result"),
                                     "Storage operation succeeded, but result was unexpected."),
        UNKNOWN_STORAGE_ERROR(STORAGE_ERROR,
                              999,
                              500,
                              "UnknownStorageError",
                              Keyword.intern("storage-error",
                                             "unknown-error"),
                              "An unknown error occurred while communicating with the backing store."),
    MESSAGING_ERROR(null,
                    Long.MIN_VALUE,
                    500,
                    "MessagingError",
                    Keyword.intern("eva.error",
                                   "messaging-error"),
                    "null"),
        MESSAGING_IO_FAILURE(MESSAGING_ERROR,
                             2000,
                             502,
                             "MessagingIOFailure",
                             Keyword.intern("messaging-error",
                                            "bad-response"),
                             "Bad response received while communicating with the Eva broker."),
        UNKNOWN_MESSAGING_ERROR(MESSAGING_ERROR,
                                2001,
                                500,
                                "UnknownMessagingError",
                                Keyword.intern("messaging-error",
                                               "unknown-error"),
                                "Unknown error occurred while communicating with the Eva broker."),
        MESSAGE_DISPATCH_FAILURE(MESSAGING_ERROR,
                                 2002,
                                 500,
                                 "MessageDispatchFailure",
                                 Keyword.intern("messaging-error",
                                                "dispatch"),
                                 "Failed to send message."),
        MESSAGE_RECEPTION_FAILURE(MESSAGING_ERROR,
                                  2003,
                                  500,
                                  "MessageReceptionFailure",
                                  Keyword.intern("messaging-error",
                                                 "reception"),
                                  "Failed to read message."),
    CONNECTION_MANAGEMENT_ERROR(null,
                                Long.MIN_VALUE,
                                500,
                                "ConnectionManagementError",
                                Keyword.intern("eva.error",
                                               "connection-management"),
                                "null"),
        TRANSACTOR_CONNECTION_ERROR(CONNECTION_MANAGEMENT_ERROR,
                                    Long.MIN_VALUE,
                                    500,
                                    "TransactorConnectionError",
                                    Keyword.intern("connection-management",
                                                   "transactor-connection-error"),
                                    "null"),
            TRANSACTOR_NOT_FOUND(TRANSACTOR_CONNECTION_ERROR,
                                 1000,
                                 500,
                                 "TransactorNotFound",
                                 Keyword.intern("connection-management",
                                                "transactor-not-found"),
                                 "Transactor discovery failed. Could not locate transactor network address."),
            TRANSACTOR_CONNECTION_FAILURE(TRANSACTOR_CONNECTION_ERROR,
                                          1001,
                                          500,
                                          "TransactorConnectionFailure",
                                          Keyword.intern("connection-management",
                                                         "connection-failure"),
                                          "Network connection to the transactor has been lost; please manually reconnect."),
        LOCAL_CONNECTION_ERROR(CONNECTION_MANAGEMENT_ERROR,
                               Long.MIN_VALUE,
                               500,
                               "LocalConnectionError",
                               Keyword.intern("connection-management",
                                              "local-connection-error"),
                               "null"),
            PEER_SHUTDOWN(LOCAL_CONNECTION_ERROR,
                          1050,
                          500,
                          "PeerShutdown",
                          Keyword.intern("connection-management",
                                         "peer-shutdown"),
                          "The peer has been shutdown and all its associated resources have been stopped."),
            CONNECTION_INACTIVE(LOCAL_CONNECTION_ERROR,
                                1100,
                                500,
                                "ConnectionInactive",
                                Keyword.intern("connection-management",
                                               "peer-connection-inactive"),
                                "The peer's connection object is not active or has been released. Please reconnect."),
            PEER_CONNECTION_FAILED(LOCAL_CONNECTION_ERROR,
                                   1120,
                                   500,
                                   "PeerConnectionFailed",
                                   Keyword.intern("connection-management",
                                                  "peer-unable-to-connect"),
                                   "The connect call was unable to acquire the requisite resources to establish a local database connection."),
        CATALOG_CONNECTION_ERROR(CONNECTION_MANAGEMENT_ERROR,
                                 Long.MIN_VALUE,
                                 500,
                                 "CatalogConnectionError",
                                 Keyword.intern("connection-management",
                                                "catalog-connection-error"),
                                 "null"),
            CATALOG_NOT_FOUND(CATALOG_CONNECTION_ERROR,
                              1200,
                              500,
                              "CatalogNotFound",
                              Keyword.intern("connection-management",
                                             "catalog-not-found"),
                              "Catalog discovery failed. Could not locate catalog network address."),
            BAD_CATALOG_RESPONSE(CATALOG_CONNECTION_ERROR,
                                 1201,
                                 502,
                                 "BadCatalogResponse",
                                 Keyword.intern("connection-management",
                                                "bad-catalog-response"),
                                 "The catalog is preaching nonsense."),
            UNKNOWN_CATALOG_ERROR(CATALOG_CONNECTION_ERROR,
                                  1202,
                                  500,
                                  "UnknownCatalogError",
                                  Keyword.intern("connection-management",
                                                 "unknown-catalog-error"),
                                  "An unknown error occurred while communicating with the catalog."),
        STORAGE_CONNECTION_ERROR(CONNECTION_MANAGEMENT_ERROR,
                                 Long.MIN_VALUE,
                                 500,
                                 "StorageConnectionError",
                                 Keyword.intern("connection-management",
                                                "storage-connection-error"),
                                 "Failed to connect to storage."),
            STORAGE_AUTH_FAILURE(STORAGE_CONNECTION_ERROR,
                                 1301,
                                 500,
                                 "StorageAuthFailure",
                                 Keyword.intern("connection-management",
                                                "storage-authentication-failure"),
                                 "Failed to authenticate with storage backend."),
    PROCESSING_FAILURE(null,
                       Long.MIN_VALUE,
                       422,
                       "ProcessingFailure",
                       Keyword.intern("eva.error",
                                      "processing-failure"),
                       "null"),
        TRANSACTION_PIPELINE_FAILURE(PROCESSING_FAILURE,
                                     Long.MIN_VALUE,
                                     422,
                                     "TransactionPipelineFailure",
                                     Keyword.intern("processing-failure",
                                                    "transaction-pipeline"),
                                     "null"),
            MODEL_CONSTRAINT_VIOLATION(TRANSACTION_PIPELINE_FAILURE,
                                       5000,
                                       422,
                                       "ModelConstraintViolation",
                                       Keyword.intern("processing-failure",
                                                      "model-constraint-violation"),
                                       "Transacting the tx-data would result in the violation of a schema constraint."),
            TRANSACTION_FUNCTION_INVALID_RETURN(TRANSACTION_PIPELINE_FAILURE,
                                                5200,
                                                422,
                                                "TransactionFunctionInvalidReturn",
                                                Keyword.intern("processing-failure",
                                                               "malformed-transaction-fn-return"),
                                                "A transaction function had a bad return value."),
            TRANSACTION_FUNCTION_EXCEPTION(TRANSACTION_PIPELINE_FAILURE,
                                           5201,
                                           422,
                                           "TransactionFunctionException",
                                           Keyword.intern("processing-failure",
                                                          "failed-transaction-fn"),
                                           "A transaction function threw an exception."),
            UNKNOWN_TRANSACTION_FAILURE(TRANSACTION_PIPELINE_FAILURE,
                                        5249,
                                        422,
                                        "UnknownTransactionFailure",
                                        Keyword.intern("processing-failure",
                                                       "unknown-error"),
                                        "Something bad happened in the transaction process."),
        DATALOG_INTERPRETER_FAILURE(PROCESSING_FAILURE,
                                    Long.MIN_VALUE,
                                    422,
                                    "DatalogInterpreterFailure",
                                    Keyword.intern("processing-failure",
                                                   "datalog-interpreter"),
                                    "null"),
            UNCOMPUTABLE_QUERY(DATALOG_INTERPRETER_FAILURE,
                               5250,
                               422,
                               "UncomputableQuery",
                               Keyword.intern("processing-failure",
                                              "uncomputable-query"),
                               "The datalog interpreter has rejected this query as uncomputable."),
            UNKNOWN_QUERY_PREDICATE(DATALOG_INTERPRETER_FAILURE,
                                    5300,
                                    422,
                                    "UnknownQueryPredicate",
                                    Keyword.intern("processing-failure",
                                                   "unknown-query-predicate"),
                                    "Symbol does not map to known built-in or rule predicate."),
            QUERY_FUNCTION_NOT_FOUND(DATALOG_INTERPRETER_FAILURE,
                                     5350,
                                     422,
                                     "QueryFunctionNotFound",
                                     Keyword.intern("processing-failure",
                                                    "query-function-not-found"),
                                     "Unable to resolve a function symbol in the query or rules."),
            QUERY_FUNCTION_INVALID_RETURN(DATALOG_INTERPRETER_FAILURE,
                                          5351,
                                          422,
                                          "QueryFunctionInvalidReturn",
                                          Keyword.intern("processing-failure",
                                                         "unparsable-query-fn-return"),
                                          "Unable to parse the return value of a function called in the query or rules."),
            QUERY_FUNCTION_FAILURE(DATALOG_INTERPRETER_FAILURE,
                                   5352,
                                   422,
                                   "QueryFunctionFailure",
                                   Keyword.intern("processing-failure",
                                                  "failed-query-fn"),
                                   "A function called in the query threw an exception."),
            INVALID_QUERY_ARGS(DATALOG_INTERPRETER_FAILURE,
                               5353,
                               422,
                               "InvalidQueryArgs",
                               Keyword.intern("processing-failure",
                                              "invalid-query-arguments"),
                               "Invalid arguments to query.");

    private EvaErrorCode parent;
    private long evaCode;
    private long httpCode;
    private String name;
    private Keyword keyword;
    private String explanation;

    // https://en.wikipedia.org/wiki/Initialization-on-demand_holder_idiom
    private static class Mapping {
        static Map<Long, EvaErrorCode> CODE_TO_CODE = new HashMap<>();
    }

    private EvaErrorCode(EvaErrorCode parent,
                         long evaCode,
                         long httpCode,
                         String name,
                         Keyword keyword,
                         String explanation) {
        this.parent = parent;
        this.evaCode = evaCode;
        this.httpCode = httpCode;
        this.name = name;
        this.keyword = keyword;
        // this.keyword = Keyword.intern(keyword);
        this.explanation = explanation;
        Mapping.CODE_TO_CODE.put(evaCode, this);
    }

    public static EvaErrorCode getFromLong(Long code) {
        return Mapping.CODE_TO_CODE.get(code);
    }

    @Override
    public boolean isUnspecified() {
        return evaCode == -1;
    }

    /*
     * -1 is used by UnknownError.
     * Long.MIN_VALUE is used by all non-terminal error codes.
     * The intention is that these error codes will never be returned directly,
     * but are used merely to provide a sensible hierarchy for `is()` calls.
    */
    @Override
    public long getCode() {
        return evaCode;
    }

    @Override
    public long getHttpErrorCode() {
        return httpCode;
    }

    @Override
    public String getScope() {
        if (parent == null) {
            return "";
        } else {
            return parent.getScope() + parent.getName() + ": ";
        }
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public Keyword getKey() {
        return keyword;
    }

    @Override
    public String getExplanation() {
        return explanation;
    }

    @Override
    public boolean is(IErrorCode e) {
        return (this == e) ? true :
               (parent == null) ? false :
               parent.is(e);
    }

}
