(ns common_foxeriot.integrant.rabbit
  (:require [integrant.core :as ig]
            [langohr.core :as rmq]
            [langohr.consumers :as lc]
            [langohr.channel :as lch]
            [langohr.queue :as lq]
            [taoensso.timbre :as log]
            [langohr.basic :as lb])
  (:import (com.rabbitmq.client AlreadyClosedException)))

;; XXX(miikka) This is probably not very good health check! It gives false alerts
;; all the time and I don't think it has ever alerted me about a real problem. :(
(defn health-check
  "Returns true if the connection works."
  [{:keys [chan]}]
  ;; We declare a queue, send a random message through it and consume it.
  (try
    (let [queue (lq/declare-server-named chan {:exclusive true})
          test-message (str (rand-int Integer/MAX_VALUE))
          result (promise)
          consumer (lc/subscribe chan queue #(deliver result (String. ^bytes %3 "UTF-8"))
                                 {:auto-ack true :no-local true})]
      (try
        (lb/publish chan "" queue test-message)
        (= test-message (deref result 2000 nil))
        (finally
          (lb/cancel chan consumer))))
    (catch AlreadyClosedException _
      false)))

(defn open-channel
  "Open a channel over an AMQP connection."
  [{:keys [conn prefetch-count]}]
  (doto (lch/open conn)
    (lb/qos prefetch-count)))

(defmethod ig/init-key :rabbit/mq [_ {:keys [conn-opts queues prefetch-count]
                                      :or {prefetch-count 100}}]
  (let [conn (do
               (log/infof "Creating a connection to RabbitMQ with connect options %s"
                          (dissoc conn-opts :username :password))
               (rmq/connect conn-opts))
        chan (open-channel {:conn conn :prefetch-count prefetch-count})]
    (doseq [[queue-name options] queues]
      (log/infof "Declaring queue %s" queue-name)
      (lq/declare chan queue-name options))
    {:conn conn :chan chan :prefetch-count prefetch-count}))

(defmethod ig/halt-key! :rabbit/mq [_ {:keys [conn chan]}]
  (when chan
    (log/infof "Closing RabbitMQ channel")
    (try
      (rmq/close chan)
      (catch AlreadyClosedException _)))
  (when conn
    (log/infof "Closing RabbitMQ connection")
    (try
      ;; RabbitMQ client's Connection is supposedly Closeable, so you might expect this close call to be idempotent.
      ;; It's not, therefore we try and catch.
      (rmq/close conn)
      (catch AlreadyClosedException _))))
