(ns clj-honeycomb.middleware.ring
  "Ring middleware to turn every request/response into a Honeycomb event."
  (:use [clojure.future])
  (:require [clojure.spec.alpha :as s]
            [clojure.string :as str]

            [clj-honeycomb.core :as honeycomb]
            [clj-honeycomb.propagation :as propagation]
            [clj-honeycomb.util.map :as map-util]))

(s/fdef get-header
  :args (s/cat :request map?
               :header string?)
  :ret string?)

(defn- get-header
  "Get a header from a Ring request map, regardless of whether or not the header
   name is given as a string or keyword or whether or not the name of the header
   is lower cased.

   request  The Ring request map
   header   The name of the header."
  [request header]
  (get (->> request
            :headers
            (map (fn [[k v]]
                   [(str/lower-case (name k)) v]))
            (into {}))
       (->> header
            name
            str/lower-case)))

(s/fdef get-headers
  :args (s/cat :request map?
               :headers (s/coll-of string?))
  :ret (s/map-of string? string?))

(defn- get-headers
  "Like get-header, but for multiple headers.

   request  The Ring request map.
   headers  A list of headers to select from the request map."
  [request headers]
  (reduce (fn [acc header]
            (when-let [v (get-header request header)]
              (assoc acc header v)))
          {}
          (map (comp str/lower-case name) headers)))

(s/fdef trace-data
  :args (s/cat :request map?)
  :ret map?)

(defn- trace-data
  "If trace data has been propagated from the requesting HTTP client, unpack
   it and use it to attach the event generated by this middleware as a child
   span for the event in the requesting HTTP client.

   request  The Ring request map."
  [request]
  (if-let [trace-header (get-header request "X-Honeycomb-Trace")]
    (try
      (or (propagation/unpack trace-header) {})
      (catch Exception _
        {}))
    {}))

(s/fdef default-extract-request-fields
  :args (s/cat :request map?)
  :ret map?)

(defn- default-extract-request-fields
  "Convert a Ring request into a map of fields to be added to the event.

   request  The Ring request map."
  [request]
  (->> {:headers (get-headers request ["Content-Length"
                                       "Host"
                                       "User-Agent"
                                       "X-Forwarded-For"
                                       "X-Forwarded-Proto"])}
       (merge (cond (:query-params request) {:params (:query-params request)}
                    (:query-string request) {:params (:query-string request)}
                    :else nil))
       (merge (select-keys request [:protocol
                                    :remote-addr
                                    :request-method
                                    :scheme
                                    :server-name
                                    :server-port
                                    :uri]))
       (map-util/flatten-and-stringify "ring.request.")
       (filter (comp some? val))
       (into {})))

(s/fdef default-extract-response-fields
  :args (s/cat :response map?)
  :ret map?)

(defn- default-extract-response-fields
  "Convert a Ring response into a map of fields to be added to the event.

   response  The Ring response map."
  [response]
  (->> {:status (:status response)
        :headers (get-headers response ["Content-Encoding"
                                        "Content-Length"
                                        "Content-Type"])}
       (map-util/flatten-and-stringify "ring.response.")
       (filter (comp some? val))
       (into {})))

(s/def ::extract-request-fields (s/with-gen (s/fspec :args (s/cat :request map?)
                                                     :ret map?)
                                  #(s/gen #{(fn [_map] {})
                                            default-extract-request-fields})))
(s/def ::extract-response-fields (s/with-gen (s/fspec :args (s/cat :response map?)
                                                      :ret map?)
                                   #(s/gen #{(fn [_map] {})
                                             default-extract-response-fields})))
(s/def ::honeycomb-event-data (s/and map? seq))
(s/def ::honeycomb-event-options :clj-honeycomb.core/send-options)
(s/def ::with-honeycomb-event-options (s/keys :opt-un [::extract-request-fields
                                                       ::extract-response-fields
                                                       ::honeycomb-event-data
                                                       ::honeycomb-event-options]))

(s/fdef ring-request->event-data
  :args (s/cat :event-data (s/nilable map?)
               :extract-request-fields ::extract-request-fields
               :request map?)
  :ret ::honeycomb-event-data)

(defn- ring-request->event-data
  [event-data extract-request-fields request]
  (let [{:keys [context parent-span-id trace-id]} (trace-data request)]
    (merge {}
           (when (and trace-id context)
             context)
           event-data
           {:name (str (-> request :request-method name str/upper-case)
                       " "
                       (:uri request))}
           (extract-request-fields request)
           {:traceId (or trace-id (honeycomb/generate-trace-id))}
           (when (and trace-id parent-span-id)
             {:parentId parent-span-id}))))

(s/fdef with-honeycomb-event
  :args (s/alt :without-options (s/cat :handler fn?)
               :with-options (s/cat :options ::with-honeycomb-event-options
                                    :handler fn?))
  :ret (s/alt :sync (s/fspec :args (s/cat :request map?)
                             :ret map?)
              :async (s/fspec :args (s/cat :request map?
                                           :respond fn?
                                           :raise fn?)
                              :ret any?)))

(defn with-honeycomb-event
  "Ring middleware to turn every request/response into a Honeycomb event. By
   default every item in the request map and the status and headers from the
   response map are included in the event. If you have sensitive data in your
   request or response maps you may wish to implement your own
   :extract-request-fields or :extract-response-fields functions to prevent them
   from leaking to Honeycomb.

   options  A map with any of the following items:

            :extract-request-fields  A function which takes one argument, the
                                     Ring request map and returns a map of data
                                     to be added to the event.
            :extract-response-fields A function which takes one argument, the
                                     Ring response map and returns a map of data
                                     to be added to the event.
            :honeycomb-event-data    Fields to be added to the event regardless
                                     of request or response. Will be overridden
                                     by anything added by extract-request-fields
                                     or extract-response-fields.
            :honeycomb-event-options Options to be passed to
                                     clj-honeycomb.core/send as the options
                                     argument. See that function for full
                                     documentation as to what's supported.
   handler  The Ring handler function."
  ([handler]
   (with-honeycomb-event {} handler))
  ([options handler]
   (let [make-event-data (partial ring-request->event-data
                                  (:honeycomb-event-data options)
                                  (or (:extract-request-fields options)
                                      default-extract-request-fields))
         add-data-from-response (fn [response]
                                  (let [f (or (:extract-response-fields options)
                                              default-extract-response-fields)]
                                    (honeycomb/add-to-event (f response)))
                                  response)
         event-options (or (:honeycomb-event-options options) {})]
     (fn
       ([request]
        (honeycomb/with-event (make-event-data request) event-options
          (add-data-from-response (handler request))))
       ([request respond raise]
        (handler request
                 (fn [response]
                   (honeycomb/with-event (make-event-data request) event-options
                     (add-data-from-response (respond response))))
                 (fn [exception]
                   (honeycomb/with-event (make-event-data request) event-options
                     (honeycomb/add-to-event {:exception exception})
                     (raise exception)))))))))
