;
;     This file is part of defenv.
;
;     defenv is free software: you can redistribute it and/or modify
;     it under the terms of the GNU General Public License as published by
;     the Free Software Foundation, either version 3 of the License, or
;     (at your option) any later version.
;
;     defenv is distributed in the hope that it will be useful,
;     but WITHOUT ANY WARRANTY; without even the implied warranty of
;     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;     GNU General Public License for more details.
;
;     You should have received a copy of the GNU General Public License
;     along with defenv.  If not, see <http://www.gnu.org/licenses/>.
;

;; # Welcome to defenv
;;
;; This library attempts to simplify the management of environment variables.
;; Put simply, it's `System/getenv` on steroids.
;;
;; Features include: documentation generation, default values, and custom
;; parsing. Also critical is the use of delayed binding so your application
;; doesn't throw exceptions when loading namespaces (a huge pet peeve!).
;;
;; Take a look at `defenv.usage` for examples of how to use the library.

(ns defenv.core
  (:require [clojure.pprint :refer :all]
            [clojure.set :refer :all]
            [slingshot.slingshot :refer [throw+]]))

(def global-defined-spec (atom (sorted-map)))

;; ## Parsing convenience functions

(defn parse-bool [s] (Boolean/parseBoolean s))
(defn parse-int [s] (Integer/parseInt s))
(defn parse-long [s] (Long/parseLong s))
(defn parse-double [s] (Double/parseDouble s))
(defn parse-float [s] (Float/parseFloat s))

(defn parse-boolean
  "DEPRECATED in favor of parse-bool"
  {:deprecated "0.0.6"}
  [s] (parse-bool s))

;; ## Test fixtures

(defn reset-defined-env! []
  (swap! global-defined-spec empty))

;; ## On-the-fly documentation generation

(declare env make-usage-string)

;; ### Display

;; "Change this to true if you want nice error tables to be printed to
;; stderr (or whatever; see print-usage)."
(def ^:dynamic print-usage? (atom false))

(defn print-usage-err
  ([]
   (print-usage-err @global-defined-spec))
  ([defined-spec]
   (binding [*out* *err*]
     (println
       (make-usage-string defined-spec "Environment variable[s] missing:")))))

(defn set-usage-print-enabled! [enabled?]
  (reset! print-usage? enabled?))

(defmacro with-usage-print-enabled [& body]
  `(binding [print-usage? (delay true)] ~@body))

;; ### Generation

(defn- get-var-status [{:keys [e-name v default masked? optional?] :as doc-map}]
  (let [e-val (env e-name default)]
    (-> doc-map
        (assoc :value
               (if e-val (if masked? ::masked @v)
                         (if optional? "nil" ::missing)))
        (rename-keys {:e-name :name}))))

(defn- get-var-statuses
  ([] (get-var-statuses @global-defined-spec))
  ([defined-spec] (map get-var-status (vals defined-spec))))

(defn- throw-usage
  ([] (throw-usage @global-defined-spec))
  ([defined-spec]
   (when @print-usage? (print-usage-err defined-spec))
   (throw+ {:type ::missing-env
            :missing (->> (get-var-statuses defined-spec)
                          (filter #(= ::missing (:value %)))
                          (map :name))})))

(def ^:private displays {::missing "*REQUIRED*"
                         ::masked "--MASKED--"})

(defn make-displayable [v]
  (if-let [displayable (displays v)] displayable v))

(defn make-usage-string
  "Display nice documentation telling folks which environment variables need
   to be set, and what the current values are. Please note that using this
   function within a main function that doesn't link to any namespaces that
   define environment variables will cause the library to tell you there are
   no environment variables defined. This is due to Clojure's lazy loading
   of namespaces."
  ([defined-spec prefix]
   (let [env-docs (map #(update % :value make-displayable)
                       (get-var-statuses defined-spec))
         doc-str (with-out-str (print-table [:name :value :doc] env-docs))]
     (if (empty? env-docs)
       "No environment variables defined!"
       (str prefix "\n" doc-str))))
  ([prefix] (make-usage-string @global-defined-spec prefix)))

;; ## Core functionality

(defn- env-or [env-name f]
  (let [val (System/getenv env-name)]
    (if-not val (f) val)))

(defn env
  "### Retrieving an environment variable on-the-fly
   If no default value is set and the variable is not present, an exception will
   be thrown displaying which variables are missing and which are set.

   Using this function directly in your code is not recommended, as users will
   not have any way of knowing they missed an environment variable. Of course,
   you could use it if you want there to be secret variables to enable various
   behaviors?"
  ([env-name] (env-or env-name throw-usage))
  ([env-name default-value] (env-or env-name #(identity default-value))))

(defn- parse-env [env-name {:keys [default tfn optional?] :as params}]
  (let [has-param? (partial contains? params)
        env-args [env-name]
        base-value (env env-name default)]
    {:tfn (if (and (not (nil? base-value)) (has-param? :tfn)) tfn identity)
     :env-args (if (or optional? (has-param? :default))
                 (conj env-args default) env-args)
     :params-to-display (assoc params :e-name env-name)
     :base-value base-value}))

(defn- add-doc [doc-present? doc-or-env params]
  (if doc-present? (assoc params :doc doc-or-env) params))

(defn- overlay-env [m k {:keys [optional?] env-name :env :as params}]
  (let [{:keys [tfn params-to-display base-value]}
        (parse-env env-name params)

        v (tfn base-value)]
    (-> m
        (update :result #(if (and (not optional?) (nil? v)) % (assoc % k v)))
        (update :display-spec assoc env-name
                (assoc params-to-display :v (delay v))))))

(defn- key-set [m] (set (keys m)))

(defn env->map
  "### Retrieving a map of environment variable values

   Operates much like `defenv` except you can define multiple bindings at once,
   and receive a map of all values that have been found in the environment.

   The map should look something like this:

       {:ev1 {:env \"MY_ENV_VAR\"
              :tfn my-optional-parse-function
              :default \"MY OPTIONAL DEFAULT VALUE\"
              :masked? true
              :doc \"Nice documentation\"
              :optional? true}
        :other {:env \"OTHER_VAR\"}}

   In this case, the `:ev1` key will be filled in with the value of
   `MY_ENV_VAR`, unless it isn't present, in which case, it will receive the
   value `MY OPTIONAL DEFAULT VALUE`. Every key except for `:env` is optional.

   In the case of `:other`, if there is no value for `OTHER_VAR`, there will be
   an exception thrown much like when attempting to deref a binding generated
   by `defenv`."
  [env-spec]
  (let [{:keys [result display-spec]}
        (reduce-kv overlay-env
                   {:result (sorted-map) :display-spec (sorted-map)} env-spec)
        missing-keys (difference (key-set env-spec) (key-set result))]
    (if (empty? missing-keys)
      result
      (throw-usage display-spec))))

(defmacro defenv
  "### Defining an environment variable binding

   Define a binding `b` to an environment variable. Creates a delayed
   object that, when dereferenced, will load an environment variable of the
   given key.

   If `doc-or-env` and `env-or-fk` are both strings, we assume that
   `doc-or-env` is a docstring and `env-or-fk` is the environment variable
   to be pulled. Then, `remaining` become the `params`.

   Else, we assume `doc-or-env` is the environment variable and
   we use `env-or-fk` as the first key of the `params`. This
   convention also allows for your documentation generator
   (like <https://github.com/gdeer81/marginalia>)
   to detect docstrings in their conventional position.

   If `:default {value}` shows up in the params, will send back the
   given value if the variable isn't defined.

   If `:tfn {function}` shows up in the params, runs the given function against
   the result of the environment variable load, which is by default a string.

   If you add a `:masked?` keyword and set it to `true`, the value won't be
   displayed in usage documentation.

   If you add a `:optional?` keyword and set it to `true`, then there need not
   be a default value set, and the `tfn` will not get invoked if the value is
   missing."
  [b & [doc-or-env env-or-fk & remaining]]
  (let [doc-present? (every? string? [doc-or-env env-or-fk])
        env-name (if doc-present? env-or-fk doc-or-env)
        params (->> (if doc-present? remaining (concat [env-or-fk] remaining))
                    (apply hash-map)
                    (add-doc doc-present? doc-or-env))
        {:keys [tfn env-args params-to-display]} (parse-env env-name params)]
    `(let [v# (delay (~tfn (env ~@env-args)))]
       (swap! global-defined-spec
              assoc ~env-name (assoc ~params-to-display :v v#))
       (def ~b v#))))