;
;     This file is part of defenv.
;
;     defenv is free software: you can redistribute it and/or modify
;     it under the terms of the GNU General Public License as published by
;     the Free Software Foundation, either version 3 of the License, or
;     (at your option) any later version.
;
;     defenv is distributed in the hope that it will be useful,
;     but WITHOUT ANY WARRANTY; without even the implied warranty of
;     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;     GNU General Public License for more details.
;
;     You should have received a copy of the GNU General Public License
;     along with defenv.  If not, see <http://www.gnu.org/licenses/>.
;

;; # Welcome to defenv
;;
;; This library attempts to simplify the management of environment variables.
;; Put simply, it's `System/getenv` on steroids.
;;
;; Features include: documentation generation, default values, and custom
;; parsing. Also critical is the use of delayed binding so your application
;; doesn't throw exceptions when loading namespaces (a huge pet peeve!).
;;
;; Take a look at `defenv.usage` for examples of how to use the library.

(ns defenv.core
  (:require [clojure.pprint :refer :all]
            [clojure.set :refer :all]
            [slingshot.slingshot :refer [throw+]]))

(defn- env
  [env-name default-value]
  (let [val (System/getenv env-name)] (if-not val default-value val)))

;; ## Transformation convenience functions
;; (for use wherever you see `:tfn`)

(defn parse-bool [s] (Boolean/parseBoolean s))
(defn parse-int [s] (Integer/parseInt s))
(defn parse-long [s] (Long/parseLong s))
(defn parse-float [s] (Float/parseFloat s))
(defn parse-double [s] (Double/parseDouble s))

(defn parse-boolean
  "DEPRECATED in favor of parse-bool"
  {:deprecated "0.0.6"} [s] (parse-bool s))

;; ## On-the-fly documentation

;; ### Generation

(defn- get-var-status [{:keys [env-name v default masked? optional?]
                        :as doc-map}]
  (let [e-val (env env-name default)]
    (-> doc-map
        (assoc :value
               (if e-val (if masked? ::masked v)
                         (if optional? "nil" ::missing)))
        (rename-keys {:env-name :name}))))

(defn- get-var-statuses [defined-spec] (map get-var-status (vals defined-spec)))

(def ^:private displays {::missing "*REQUIRED*" ::masked "--MASKED--"})

(defn- make-usage-string [defined-spec prefix]
  (let [convert #(if-let [displayable (displays %)] displayable %)
        display #(update % :value convert)
        env-docs (map display (get-var-statuses defined-spec))
        doc-str (with-out-str (print-table [:name :value :doc] env-docs))]
    (if (empty? env-docs)
      "No environment variables defined!"
      (str prefix "\n" doc-str))))

;; ### Display

(def ^:dynamic ^:private print-usage? (atom false))

(defn set-usage-print-enabled!
  "Use this function to enable user-friendly error printing globally."
  [enabled?]
  (reset! print-usage? enabled?))

(defmacro with-usage-print-enabled
  "Use this one to enable user-friendly error printing in the current thread."
  [& body]
  `(binding [print-usage? (delay true)] ~@body))

(defn- print-usage-err [defined-spec]
  (binding [*out* *err*]
    (println
      (make-usage-string defined-spec "Environment variable[s] missing:"))))

(defn- throw-usage-if-missing [defined-spec]
  (let [missing-vals (->> (get-var-statuses defined-spec)
                          (filter #(= ::missing (:value %)))
                          (map :name))]
    (when (seq missing-vals)
      (when @print-usage? (print-usage-err defined-spec))
      (throw+ {:type ::missing-env
               :missing missing-vals}))))

;; ## Core functionality

;; ### Retrieving a map of environment variable values

(defn- parse-env [env-name {:keys [default tfn optional?] :as params}]
  (let [has-param? (partial contains? params)
        env-args [env-name]
        base-value (env env-name default)]
    {:tfn (if (and (not (nil? base-value)) (has-param? :tfn)) tfn identity)
     :env-args (if (or optional? (has-param? :default))
                 (conj env-args default) env-args)
     :params-to-display (assoc params :env-name env-name)
     :base-value base-value}))

(defn- overlay-env [m k {:keys [optional?] env-name :env :as params}]
  (let [{:keys [tfn params-to-display base-value]} (parse-env env-name params)
        v (tfn base-value)]
    (-> m
        (update :env-map #(if (and (not optional?) (nil? v)) % (assoc % k v)))
        (update :display-spec assoc env-name (assoc params-to-display :v v)))))

(def ^:private get-env-info
  (partial reduce-kv overlay-env
           {:env-map (sorted-map) :display-spec (sorted-map)}))

(defn env->map
  "Operates much like `defenv` except you can define multiple bindings at once,
   and receive a map of all values that have been found in the environment.
   Unlike `defenv`, however, this function will throw an exception if any
   required variable is missing, not just the one you ask for.

   The map should look something like this:

       {:ev1 {:env \"MY_ENV_VAR\"
              :tfn my-optional-parse-function
              :default \"MY OPTIONAL DEFAULT VALUE\"
              :masked? true
              :doc \"Nice documentation\"
              :optional? true}
        :other {:env \"OTHER_VAR\"}}

   In this case, the `:ev1` key will be filled in with the value of
   `MY_ENV_VAR`, unless it isn't present, in which case, it will receive the
   value `MY OPTIONAL DEFAULT VALUE`. Every key except for `:env` is optional.

   In the case of `:other`, if there is no value for `OTHER_VAR`, there will be
   an exception thrown much like when attempting to deref a binding generated
   by `defenv` that is required but missing."
  [env-spec]
  (let [{:keys [env-map display-spec]} (get-env-info env-spec)]
    (throw-usage-if-missing display-spec)
    env-map))

;; ### Defining a global environment variable binding

(def ^:private global-defined-spec (ref {}))
(def ^:private global-parsed-env (ref {}))
(def ^:private global-display-spec (ref (sorted-map)))

(defmacro ^:private with-new-parsed-env! [& body]
  `(dosync ~@body (alter global-parsed-env empty)))

(defn- initialize-global! []
  (when (empty? @global-parsed-env)
    (let [{:keys [env-map display-spec]} (get-env-info @global-defined-spec)]
      (dosync
        (ref-set global-parsed-env env-map)
        (ref-set global-display-spec display-spec)))))

(defmacro ^:private guarantee-global! [& body]
  `(do (initialize-global!) ~@body))

(defn reset-defined-env!
  "## Test fixture"
  []
  (with-new-parsed-env!
    (alter global-display-spec empty)
    (alter global-defined-spec empty)))

(defn- add-doc [doc-present? doc-or-env params]
  (if doc-present? (assoc params :doc doc-or-env) params))

(defn get-global-env
  "Used primarily by `defenv` to retrieve the global environment state."
  [env-name]
  (let [[parsed-env display-spec]
        (guarantee-global! [@global-parsed-env @global-display-spec])]
    (if ((set (keys parsed-env)) env-name)
      (get parsed-env env-name)
      (throw-usage-if-missing display-spec))))

(defn add-to-global-defined-spec!
  "Used primarily by `defenv` to add to the global environment state."
  [env-name params]
  (with-new-parsed-env! (alter global-defined-spec assoc env-name params)))

(defmacro defenv
  "Define a binding `b` to an environment variable. Creates a delayed
   object that, when dereferenced, will load an environment variable of the
   given key.

   If `doc-or-env` and `env-or-fk` are both strings, we assume that
   `doc-or-env` is a docstring and `env-or-fk` is the environment variable
   to be pulled. Then, `remaining` become the `params`.

   Else, we assume `doc-or-env` is the environment variable and
   we use `env-or-fk` as the first key of the `params`. This
   convention also allows for your documentation generator
   (like <https://github.com/gdeer81/marginalia>)
   to detect docstrings in their conventional position.

   If `:default {value}` shows up in the params, will send back the
   given value if the variable isn't defined.

   If `:tfn {function}` shows up in the params, runs the given function against
   the result of the environment variable load, which is by default a string.

   If you add a `:masked?` keyword and set it to `true`, the value won't be
   displayed in usage documentation.

   If you add a `:optional?` keyword and set it to `true`, then there need not
   be a default value set, and the `tfn` will not get invoked if the value is
   missing."
  [b & [doc-or-env env-or-fk & remaining]]
  (let [doc-present? (every? string? [doc-or-env env-or-fk])
        env-name (if doc-present? env-or-fk doc-or-env)
        params (->> (if doc-present? remaining (concat [env-or-fk] remaining))
                    (apply hash-map)
                    (add-doc doc-present? doc-or-env))
        params (assoc params :env env-name)]
    `(do (add-to-global-defined-spec! ~env-name ~params)
         (def ~b (delay (get-global-env ~env-name))))))