;
; Copyright (c) 2019 Stephen Starkey.
;
; This file is part of itl.
;
; itl is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; itl is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with itl.  If not, see <http://www.gnu.org/licenses/>.
;

(ns itl.example
  (:require [itl.core :refer :all])
  (:import (java.util UUID)))

;; # itl example
;;
;; This is where we bind the functions that test itl itself.

;; ## complete-example.adoc fixtures

(defn- calc-balance [{:keys [amount balance] :as page-state}]
  (let [balance (Integer/parseInt balance)
        to-add (Integer/parseInt amount)
        actual-balance (+ balance to-add)]
    (assoc page-state :balance (str actual-balance))))

(deftfn "check register" [page-state table-data]
  (column-table page-state table-data {:assign {"Amount" :amount}
                                       :exec calc-balance
                                       :asserts {"Balance" :balance}}))

(defn assign-values [{:keys [val] :as page-state}]
  (first (generated-var page-state val)))

(deftfn "set some values" [page-state table-data]
  (column-table page-state table-data {:assign {"Value" :val}
                                       :exec assign-values
                                       :asserts {}}))

(defn check-uuids [{:keys [val] :as page-state}]
  (try
    (UUID/fromString val)
    (assoc page-state
      :is-uuid? "yes"
      :gen-val val)
    (catch IllegalArgumentException _
      (-> page-state (assoc :is-uuid? "no") (dissoc :gen-val)))))

(deftfn "check for uuids" [page-state table-data]
  (column-table page-state table-data {:assign {"Value" :val}
                                       :exec check-uuids
                                       :asserts {"Is UUID?" :is-uuid?
                                                 "Generated" :gen-val}}))

(defn- multiply [{:keys [x y] :as page-state}]
  (assoc page-state :result (* x y)))

(deftafn "times tables" [page-state table-data {:strs [rows]}]
  (generative-table page-state table-data
                    {:generate {"X" [:x random-integer]
                                "Y" [:y random-integer]}
                     :exec multiply
                     :asserts {"Result" [:result integer-comparison]}}
                    (Integer/parseInt rows)))

(defn- concatenate [{:keys [n prefix] :as page-state}]
  (assoc page-state
    :result (str prefix n)
    :c (str "concatenated " n " and " prefix)))

(deftafn "concatenator" [page-state table-data {:strs [rows]}]
  (generative-table page-state table-data
                    {:generate {"N" [:n random-integer]
                                "P" [:prefix loop-of-strings]
                                "Comment" [:c identity]}
                     :exec concatenate
                     :asserts {"Result" [:result bit-bob-comparison]}}
                    (Integer/parseInt rows)))

(defop "break" [s] (throw (ex-info "This op is broke" {})) s)

(deftfn "broken register" [page-state table-data]
  (column-table page-state table-data
                {:assign {"Amount" :amount "Payee" :payee}
                 :exec (fn [{:keys [payee] :as page-state}]
                         (when (= "tesla" payee)
                           (throw (ex-info "Tesla is overpriced" {})))
                         (calc-balance page-state))
                 :asserts {"Balance" :balance}}))

(deftfn "broken name list" [_ _]
  (throw (ex-info "No idea what to do here" {})))