(ns customer.core
  (:require [clojure.tools.logging :as log]
            [clojure.data.json :as json]
            [constants.core :as constants]
            [policies.core :as policies]
            [clj-time.format :as time]
            [http.core :as http]
            [cacheing.core :as cache]))

(defn- convert-json-to-map
  [json-string]
  (try
    (json/read-str json-string :key-fn keyword)
    (catch Exception e
      (log/error "Failed to parse JSON: " json-string)
      {:customers []})))

(defn- generate-request
  [mdm-id]
  (json/write-str {:searchRequest {:requestParam {:mdmId mdm-id} :header {:channelType "DSU"}}}))

(defn- call-search-api
  [body]
  (http/http-call constants/search-url body :post))

(defn search
  "Calls EADS and does a search for customers information based of MDM ID."
  [mdm-id]
  (let [request (generate-request mdm-id)
        response (or (cache/get-cache mdm-id) (call-search-api request))
        json (convert-json-to-map response)
        customer (first (:customers json))]
    (if (nil? customer)
      (log/error "Failed to find customer for " mdm-id)
      (do
        (cache/set-cache mdm-id response)
        customer))))

(def ^:private birthday-formatter
  (time/formatter "YYYY/mm/dd"))

(defn- dob-formatter
  [customers]
  (when-let [dob (:dob customers)]
    (str (time/parse birthday-formatter dob))))

;; extract to helper lib
(defn- get-set
  [type key coll]
  (first (filter (fn [item] (= type (key item))) coll)))

(defn- phones-model
  [phones]
  (let [home-phone (:phoneFullNumber (get-set "homephonenumber" :phoneType phones))
        work-phone (:phoneFullNumber (get-set "workphonenumber" :phoneType phones))
        primary-phone (:phoneFullNumber (get-set "primaryphonenumber" :phoneType phones))]
    {:home home-phone :work work-phone :primary primary-phone}))

(defn- personal-info-model
  [customers]
  {:driversLicense (:driverLicense customers)
   :email (:emailId customers)
   :maritalStatus (:maritalStatus customers)
   :dateOfBirth (dob-formatter customers)
   :gender (:gender customers)})

(defn- names-model
  [customers]
  (let [full-name (:fullName customers)
        first-name (:firstName customers)
        last-name (:lastName customers)
        middle-name (:middleName customers)]
    {:fullName full-name :firstName first-name :lastName last-name :middleName middle-name}))

(defn customer-model
  "Calls EADS for customer information and populates the model.
  **See** [Data Keys page for :customer](/data-keys#customer)"
  [customers]
  (let [personal-info (personal-info-model customers)]
      {:name (names-model customers)
       :gender (:gender personal-info)
       :email (:email personal-info)
       :dateOfBirth (:dateOfBirth personal-info)
       :driversLicense (:driversLicense personal-info)
       :phones (phones-model (:phoneList customers))
       :addresses {:home (#'policies/address-model (get-set "home" :addressType (:partyAdresses customers)))
                   :mailing (#'policies/address-model (get-set "mailing" :addressType (:partyAdresses customers)))}
       :maritalStatus (:maritalStatus personal-info)}))
