(ns identity-middleware.core
  (:require [clojure.tools.logging :as log]
            [clojure.data.json :as json]
            [auth-utils.core :as auth-utils]
            [identity-middleware.jwt-filter :as jwt]))

(defn- add-header
  [request header-name header-value]
  (assoc-in request [:headers] (merge (:headers request) {header-name header-value})))

(defn- remove-401
  [request]
  (update-in request [:headers] dissoc "x-digital-401"))

(defn- auth-failed-filter
  [request]
  (add-header request "x-digital-401" true))

(defn- add-user-or-app-id-header
  [request app-or-user]
  (let [user (:user app-or-user)
        app (:app app-or-user)]
    (cond
      (not (nil? user)) (add-header request "x-digital-user-id" (:id user))
      (not (nil? app)) (add-header request "x-digital-app-id" (:id app))
      :else
        (add-header request "x-digital-user-id" (or (:mdmid app-or-user) (:subject app-or-user))))))

(defn- add-auth-header
  [request app-or-user]
  (let [id (or (:mdmid app-or-user) (:subject app-or-user))
        data (merge {:id id} app-or-user)]
    (cond
      (jwt/has-auth? request) request
      (not (nil? (:user app-or-user))) (add-header request "x-digital-auth" (auth-utils/build-user-jwt id data))
      (not (nil? (:app app-or-user))) (add-header request "x-digital-auth" (auth-utils/build-app-jwt id data))
      :else
        (add-header request "x-digital-auth" (auth-utils/build-user-jwt id data)))))

(defn- process-successful-request
  [request app-or-user]
  (-> request remove-401 (add-auth-header app-or-user) (add-user-or-app-id-header app-or-user)))

(defn- is-authenticated?
  [request]
  (auth-utils/valid-jwt? (jwt/get-jwt-from-request request)))

(defn respond-with-invalid-jwt
  []
  {:status 401
   :body (json/write-str {:message "Unauthenticated: Digital JWT did not meet validation criteria."})
   :headers {"Content-Type" "application/json"}})

(defn respond-with-no-jwt
  []
  {:status 401
   :body (json/write-str {:message "Unauthenticated: No Digital Connect Auth (JWT)."})
   :headers {"Content-Type" "application/json"}})

(defn filter-and-process
  [handler request]
  (let [filtered-request (auth-failed-filter request)
        user (jwt/get-user-by-digital-jwt request)]
    (if (is-authenticated? filtered-request)
      (handler (process-successful-request filtered-request user))
      (respond-with-invalid-jwt))))

(defn wrap-identity-middleware-service
  "Identity middleware for Connect services.
  Processes incoming requets by assigning them the x-digital-401 header, then processing the x-digital-auth header for a valid JWT. If one exists removes x-digital-401 header and call is a success. Otherwise returns status of 401."
  [handler]
  (fn [request]
    (if (jwt/has-auth? request)
      (filter-and-process handler request)
      (respond-with-no-jwt))))