(ns customer.core
  (:require [clojure.tools.logging :as log]
            [clojure.data.json :as json]
            [constants.core :as constants]
            [policies.core :as policies]
            [clj-time.format :as time]
            [http.core :as http]
            [cacheing.core :as cache]
            [soa.core :as soa]
            [clostache.parser :as template]))

(defn- convert-json-to-map
  [json-string]
  (try
    (json/read-str json-string :key-fn keyword)
    (catch Exception e
      (log/error "Failed to parse JSON: " json-string)
      {:customers []})))

(defn- generate-request
  [mdm-id]
  (json/write-str {:searchRequest {:requestParam {:mdmId mdm-id} :header {:channelType "DSU"}}}))

(defn- call-search-api
  [body]
  (http/http-call constants/search-url body :post))

(defn search
  "Calls EADS and does a search for customers information based of MDM ID."
  [mdm-id]
  (let [request (generate-request mdm-id)
        response (or (cache/get-cache mdm-id) (call-search-api request))
        json (convert-json-to-map response)
        customer (first (:customers json))]
    (if (nil? customer)
      (log/error "Failed to find customer for " mdm-id)
      (do
        (cache/set-cache mdm-id response)
        customer))))

(defn customer-soa-error
  "Provides an error response so dev can better trouble shoot issue:\n
  ```{:customerKey custkey :error 'Customer not found from RetrieveCustomerDetails.'
   :envoyUrl <the envoy url>
   :soaUrl 'SOA url where failure occured'}```"
  [mdm-id]
  (log/error "Failed to find customer from SOA." mdm-id)
  (log/error (str "SOA URL " constants/customer-url))
  {:mdm_id mdm-id
   :error "Customer not found from RetrieveCustomerDetails."
   :soaUrl constants/customer-url})

(defn get-policy-list
  [customer]
  (let [roles (:roles customer)]
    (map (fn [role]
           (let [policy (:policy (:role role))]
                {:policyNumber (:policyNumber policy)
                 :prodTypeCode (:type policy)
                 :sourceSystem (:dataSource policy)
                 :ubiIndicator (:UBIIndicator policy)}))
         roles)))

(defn create-model
  [customer-xml]
  (let [customer (:customerSummary customer-xml)]
    {:firstName (:firstName (:preferredName customer))
     :lastName (:lastName (:preferredName customer))
     :fullName (:fullName (:preferredName customer))
     :gender (:gender customer)
     :dob (:dateOfBirth customer)
     :driversLicense (:licenseNumber (:drivingLicense customer))
     :maritalStatus (:maritalStatus customer)
     :emailId (:emailAddress (:preferredEmail customer))
     :memNum (for [x (:roles customer-xml) :when (= "PRIMARY MEMBER" (first (:roles (:role x))))] (:memberNumber x))
     :memDt (for [x (:roles customer-xml) :when (= "PRIMARY MEMBER" (first (:roles (:role x))))] (:inceptionDate x))
     :policyList (get-policy-list customer-xml)
     :customerKey (:customerRegistrationID customer)
     :mdmId (:enterpriseCustomerID customer)}))

(defn get-by-rcd
  [mdm-id]
  (let [cache-key (str mdm-id "-rcd-data")
        rcd-request (template/render-resource "templates/RetrieveCustomerDetails.xml" {:mdm-id {:id mdm-id}})
        xml (or (cache/get-cache cache-key) (soa/soa-call constants/customer-url rcd-request))
        xml-doc (soa/parse-xml xml)]
    (if (or (soa/xml-is-fault xml) (nil? xml-doc))
      (customer-soa-error mdm-id)
      (do
        (cache/set-cache cache-key xml)
        (create-model (:enterpriseCustomer (:retrieveCustomerDetailsResponse (soa/keywordize-xml xml-doc))))))))

;; extract to helper lib
(defn- get-set
  [type key coll]
  (first (filter (fn [item] (= type (key item))) coll)))

(defn- phones-model
  [phones]
  (let [home-phone (:phoneFullNumber (get-set "homephonenumber" :phoneType phones))
        work-phone (:phoneFullNumber (get-set "workphonenumber" :phoneType phones))
        primary-phone (:phoneFullNumber (get-set "primaryphonenumber" :phoneType phones))]
    {:home home-phone :work work-phone :primary primary-phone}))

(defn- personal-info-model
  [customers]
  {:driversLicense (:driverLicense customers)
   :email (:emailId customers)
   :maritalStatus (:maritalStatus customers)
   :dateOfBirth (:dob customers)
   :gender (:gender customers)})

(defn- names-model
  [customers]
  (let [full-name (:fullName customers)
        first-name (:firstName customers)
        last-name (:lastName customers)
        middle-name (:middleName customers)]
    {:fullName full-name :firstName first-name :lastName last-name :middleName middle-name}))

(defn customer-model
  "Calls EADS for customer information and populates the model.
  **See** [Data Keys page for :customer](/data-keys#customer)"
  [customers]
  (let [personal-info (personal-info-model customers)]
      {:name (names-model customers)
       :ubiEnabled (> (count (filter :ubiIndicator (:policyList customers))) 0)
       :customerKey (:customerKey customers)
       :gender (:gender personal-info)
       :email (:email personal-info)
       :dateOfBirth (:dateOfBirth personal-info)
       :driversLicense (:driversLicense personal-info)
       :phones (phones-model (:phoneList customers))
       :addresses {:home (#'policies/address-model (get-set "home" :addressType (:partyAdresses customers)))
                   :mailing (#'policies/address-model (get-set "mailing" :addressType (:partyAdresses customers)))}
       :maritalStatus (:maritalStatus personal-info)}))
