(ns framework.core
  (:require [compojure.core :refer :all :as compojure]
            [compojure.route :as route]
            [ring.middleware.cors :refer [wrap-cors]]
            [clojure.data.json :as json]
            [customer.core :as customers]
            [policies.core :as policies]
            [identity-middleware.core :as identity]
            [auth-utils.core :as auth-utils]
            [claims.core :as claims]
            [coverages.core :as coverages]
            [documents.core :as documents]
            [discounts.core :as discounts]
            [billing-history.core :as billing-history]
            [billing-detail.core :as billing-detail]))
       
(def ^:private not-found-response {:status 404 :body {:error "Not Found"} :headers {"x-digital-error" "The request resulted in a nil return value."}})

(defn auth-cookie
  [request]
  (str "x-digital-auth=" (get (:headers request) "x-digital-auth")))

(defn- execute-body-with
  ([request body]
   (let [response (body request)]
     (if (nil? response)
       not-found-response
       (merge-with clojure.set/union response {:headers {"Set-Cookie" (auth-cookie request)}}))))
  
  ([request body data]
   (let [response (body data)]
     (if (nil? response)
       not-found-response
       {:status 200 :body (json/write-str response) :headers {"Content-Type" "application/json" "Set-Cookie" (auth-cookie request)}}))))

(defn- user-id
  [request]
  (get(:headers request) "x-digital-user-id"))

(defn- supply-data
  [request data]
  (let [customer (customers/search (user-id request))
        data-loaded (map          
                     (fn [flag]
                        (case flag
                          :customer {:customer (customers/customer-model customer)}
                          :policies {:policies (policies/policies-model customer)}
                          :coverages {:coverages (coverages/coverages-model customer)}
                          :claims {:claims (claims/claims-model customer)}
                          :documents {:documents (documents/documents-model customer)}
                          :discounts {:discounts (discounts/discounts-model customer)}
                          :billing-history {:billingHistory (billing-history/billing-history-model customer)}
                          :billing-detail {:billingDetail (billing-detail/billing-detail-model customer)}
                          {:error (str "Unknown data key: " (name flag))}))
                     data)]
    (into {} data-loaded)))

(defmacro endpoint
  (; ANY endpoint for routes
   [path body]
   `(seq [(compojure/GET ~path request# (#'execute-body-with request# ~body))
          (compojure/POST ~path request# (#'execute-body-with request# ~body))]))

  (; endpoint for routes by method
   [method path body]
   (if (= method :get)
     `(compojure/GET ~path request# (#'execute-body-with request# ~body))
     `(compojure/POST ~path request# (#'execute-body-with request# ~body))))

  (; endpoint for services
   [method path data & body]
   (if (= method :get)
     `(compojure/GET ~path request#
                     (#'execute-body-with request# ~@body (#'supply-data request# ~data)))
     `(compojure/POST ~path request#
                      (#'execute-body-with request# ~@body (#'supply-data request# ~data))))))

(defn service
  [& end-points]
  (let [routes (apply compojure/routes end-points)]
    (wrap-cors (identity/wrap-identity-middleware-service routes)
               :access-control-allow-origin [#".*"]
               :access-control-allow-headers ["cookie" "x-digital-auth"]
               :access-control-allow-credentials ["true"]
               :access-control-allow-methods [:get :put :post :delete :options])))

(defn route
  [redirect-url & end-points]
  (let [flattened (flatten end-points)
        routes (apply compojure/routes flattened)]
    (wrap-cors (identity/wrap-identity-middleware-route routes redirect-url)
               :access-control-allow-origin [#".*"]
               :access-control-allow-credentials ["true"]
               :access-control-allow-headers ["cookie" "x-digital-auth"]
               :access-control-allow-methods [:get :post :options])))
