(ns cubicrobotics.ragtime-alia.core
  "Functions for loading cassandra migrations based on alia and applying them to a cassandra database."
  (:refer-clojure :exclude [load-file])
  (:require [clojure.edn :as edn]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [ragtime.protocols :as p]
            [qbits.alia :as alia]
            [clj-time.local :as l]
            [resauce.core :as resauce])
  (:use qbits.hayt
        qbits.hayt.codec.joda-time
        )
  (:import [java.io File]))

(defn- migrations-table-ddl [table-name]
  (create-table table-name
    (if-not-exists)
    (column-definitions [[:id :varchar]
                         [:created_at :timestamp]
                         [:primary-key :id]])))

(defn- ensure-migrations-table-exists [session migrations-table]
  (alia/execute session (migrations-table-ddl migrations-table)))

(defrecord AliaCassandraDatabase [session migrations-table]
  p/DataStore
  (add-migration-id [_ id]
    (ensure-migrations-table-exists session migrations-table)

    (alia/execute session (insert migrations-table (values [[:id (str id)]
                                                            [:created_at (l/local-now)] ;l/local-now ?
                                                            ]))))

  (remove-migration-id [_ id]
                       (ensure-migrations-table-exists session migrations-table)
                       (alia/execute session (delete migrations-table (where {:id id}))))

  (applied-migration-ids [_]
    (ensure-migrations-table-exists session migrations-table)
    (->> (alia/execute session (select migrations-table))
         (map :id)
         sort
         vec)))

(alter-meta! #'->AliaCassandraDatabase assoc :no-doc true)
(alter-meta! #'map->AliaCassandraDatabase assoc :no-doc true)

(defn alia-database
  "Given a session and a map of options, return a Migratable database.
  The following options are allowed:

  :migrations-table - the name of the table to store the applied migrations
                      (defaults to ragtime_migrations)"
  ([session]
   (alia-database session {}))
  ([session options]
   (->AliaCassandraDatabase session (:migrations-table options "ragtime_migrations"))))

(defrecord AliaCassandraMigration [id up down]
  p/Migration
  (id [_] id)
  (run-up!   [_ db] (doseq [l up] (alia/execute (:session db) l)))
  (run-down! [_ db] (doseq [l down] (alia/execute (:session db) l))))

(alter-meta! #'->AliaCassandraMigration assoc :no-doc true)
(alter-meta! #'map->AliaCassandraMigration assoc :no-doc true)

(defn alia-migration
  "Create a Ragtime migration from a map with a unique :id, and :up and :down
  keys that map to ordered collection of CQL strings."
  [migration-map]
  (map->AliaCassandraMigration migration-map))

(defn- file-extension [file]
  (re-find #"\.[^.]*$" (str file)))

(let [pattern (re-pattern (str "([^" File/separator "]*)" File/separator "?$"))]
  (defn- basename [file]
    (second (re-find pattern (str file)))))

(defn- remove-extension [file]
  (second (re-matches #"(.*)\.[^.]*" (str file))))

(defmulti load-files
  "Given an collection of files with the same extension, return a ordered
  collection of migrations. Dispatches on extension (e.g. \".edn\"). Extend
  this multimethod to support new formats for specifying CQL migrations."
  (fn [files] (file-extension (first files))))

(defmethod load-files :default [files])

(defmethod load-files ".edn" [files]
  (for [file files]
    (-> (slurp file)
        (edn/read-string)
        (update-in [:id] #(or % (-> file basename remove-extension)))
        (alia-migration))))

(defn- cql-file-parts [file]
  (rest (re-matches #"(.*?)\.(up|down)(?:\.(\d+))?\.cql" (str file))))

(defn- read-cql [file]
  (str/split (slurp file) #"(?m)\n\s*--;;\s*\n"))

(defmethod load-files ".cql" [files]
  (for [[id files] (group-by (comp first cql-file-parts) files)]
    (let [{:strs [up down]} (group-by (comp second cql-file-parts) files)]
      (alia-migration
       {:id   (basename id)
        :up   (vec (mapcat read-cql (sort-by str up)))
        :down (vec (mapcat read-cql (sort-by str down)))}))))

(defn- load-all-files [files]
  (->> (sort-by str files)
       (group-by file-extension)
       (vals)
       (mapcat load-files)))

(defn load-directory
  "Load a collection of Ragtime migrations from a directory."
  [path]
  (load-all-files (file-seq (io/file path))))

(defn load-resources
  "Load a collection of Ragtime migrations from a classpath prefix."
  [path]
  (load-all-files (resauce/resource-dir path)))
