(ns ragtime.jdbc
  "Functions for loading SQL migrations and applying them to a SQL database."
  (:refer-clojure :exclude [load-file])
  (:require [clojure.edn :as edn]
            [clojure.java.io :as io]
            [next.jdbc :as jdbc]
            [next.jdbc.default-options]
            [next.jdbc.result-set :as rs]
            [next.jdbc.sql :as sql]
            [clojure.string :as str]
            [ragtime.protocols :as p]
            [resauce.core :as resauce])
  (:import [java.io File] 
           [java.sql Connection SQLSyntaxErrorException]
           [java.text SimpleDateFormat]
           [java.util Date]))

(defn- get-table-metadata* [^Connection conn opts]
  (-> (.getMetaData conn)
      ;; return a java.sql.ResultSet describing all tables:
      (.getTables (.getCatalog conn) nil nil (into-array ["TABLE"]))
      (rs/datafiable-result-set conn opts)))

(defn- get-table-metadata [ds]
  (cond 
    (instance? java.sql.Connection ds)
    (get-table-metadata* ds nil)

    (next.jdbc.default-options/wrapped-connection? ds)
    (get-table-metadata* (:connectable ds) (:options ds))

    :else
    (with-open [conn (jdbc/get-connection ds)]
      (get-table-metadata* conn (:options ds)))))

(defn- metadata-matches-table? [^String table-name {:keys [table_schem table_name]}]
  (.equalsIgnoreCase table-name
                     (if (.contains table-name ".")
                       (str table_schem "." table_name)
                       table_name)))

(defn- table-exists? [ds ^String table-name]
  (some (partial metadata-matches-table? table-name)
        (get-table-metadata ds)))

(defn- ensure-migrations-table-exists [ds migrations-table]
  (when-not (table-exists? ds migrations-table)
    (let [sql (str "create table " migrations-table " (id varchar(255), created_at varchar(32))")]
      (try
        (jdbc/execute! ds [sql])
        (catch SQLSyntaxErrorException e
          (throw (SQLSyntaxErrorException.
                  ;; do not include the cause so the REPL shows this error message w/ its helpful tip
                  (str "Failed to create the migrations table due to syntax incompatibilities."
                       " Cause: " (ex-message e)
                       " You need to create it yourself, something like: "
                       sql))))))))

(defn- format-datetime [^Date dt]
  (-> (SimpleDateFormat. "yyyy-MM-dd'T'HH:mm:ss.SSS")
      (.format dt)))

(defrecord SqlDatabase [ds migrations-table]
  p/DataStore
  (add-migration-id [_ id]
    (ensure-migrations-table-exists ds migrations-table)
    (sql/insert! ds migrations-table
                 {:id         (str id)
                  :created_at (format-datetime (Date.))}))

  (remove-migration-id [_ id]
    (ensure-migrations-table-exists ds migrations-table)
    (sql/delete! ds migrations-table ["id = ?" id]))

  (applied-migration-ids [_]
    (ensure-migrations-table-exists ds migrations-table)
    (->> (sql/query ds
               [(str "SELECT id FROM " migrations-table " ORDER BY created_at")])
         (map :id))))

(defn sql-database
  "Given a ds and a map of options, return a Migratable database.
  The following options are allowed:

  :migrations-table - the name of the table to store the applied migrations
                      (defaults to ragtime_migrations)"
  ([ds]
   (sql-database ds {}))
  ([ds options]
   (->SqlDatabase 
    (jdbc/with-options 
     ds
     (merge (:options ds) {:builder-fn rs/as-unqualified-lower-maps})) 
    (:migrations-table options "ragtime_migrations"))))

(defn- execute-sql! [ds statements transaction?]
  (if transaction?
    (jdbc/with-transaction [tx ds]
      (doseq [s statements]
        (jdbc/execute! (jdbc/with-options tx (:options ds)) [s])))
    (doseq [s statements]
      (jdbc/execute! ds [s]))))

(defrecord SqlMigration [id up down transactions]
  p/Migration
  (id [_] id)
  (run-up! [_ db]
    (execute-sql! (:ds db)
                  up
                  (contains? #{:up :both true} transactions)))
  (run-down! [_ db]
    (execute-sql! (:ds db)
                  down
                  (contains? #{:down :both true} transactions))))

(defn sql-migration
  "Create a Ragtime migration from a map with a unique :id, and :up and :down
  keys that map to ordered collection of SQL strings."
  [migration-map]
  (map->SqlMigration migration-map))

(defn- file-extension [file]
  (re-find #"\.[^.]*$" (str file)))

(let [pattern (re-pattern (str "([^\\/]*)\\/?$"))]
  (defn- basename [file]
    (second (re-find pattern (str file)))))

(defn- remove-extension [file]
  (second (re-matches #"(.*)\.[^.]*" (str file))))

(defmulti load-files
  "Given an collection of files with the same extension, return a ordered
  collection of migrations. Dispatches on extension (e.g. \".edn\"). Extend
  this multimethod to support new formats for specifying SQL migrations."
  (fn [files] (file-extension (first files))))

(defmethod load-files :default [files])

(defmethod load-files ".edn" [files]
  (for [file files]
    (-> (slurp file)
        (edn/read-string)
        (update-in [:id] #(or % (-> file basename remove-extension)))
        (update-in [:transactions] (fnil identity :both))
        (sql-migration))))

(defn- sql-file-parts [file]
  (rest (re-matches #"(.*?)\.(up|down)(?:\.(\d+))?\.sql" (str file))))

(defn- read-sql [file]
  (str/split (slurp file) #"(?m)\n\s*--;;\s*\n"))

(defmethod load-files ".sql" [files]
  (for [[id files] (->> files
                        (group-by (comp first sql-file-parts))
                        (sort-by key))]
    (let [{:strs [up down]} (group-by (comp second sql-file-parts) files)]
      (sql-migration
       {:id   (basename id)
        :up   (vec (mapcat read-sql (sort-by str up)))
        :down (vec (mapcat read-sql (sort-by str down)))}))))

(defn- load-all-files [files]
  (->> (group-by file-extension files)
       (vals)
       (mapcat load-files)
       (sort-by :id)))

(defn load-directory
  "Load a collection of Ragtime migrations from a directory."
  [path]
  (load-all-files (map #(.toURI ^File %) (file-seq (io/file path)))))

(defn load-resources
  "Load a collection of Ragtime migrations from a classpath prefix."
  [path]
  (load-all-files (resauce/resource-dir path)))
