(ns ingestion.api.client.dataflow-mapping-gen-api
  (:require [clojure.java.io :as io]
            [ksql.gen.core-error-msg :as emsg]
            [ksql.gen.macro.metadata-crawler-impl :as mcrawler]
            [ksql.gen.reader.xmltojson :as xmltojson]
            [clojure.xml :as xml]
            [clojure.walk :as w]
            [cheshire.core :as json]
            [ksql.gen.protocol :as p]
            [ksql.gen.reader.file-reader :as r]
            [ingestion.api.client.ns-repo-impl :as core])
  (:import (java.util.zip ZipInputStream ZipFile ZipEntry)))


(defn get-header [in]
  (with-open [is (io/reader in)]
    (let [w (first (line-seq is))
          w1 (clojure.string/split w #"\;")
          w2 (clojure.string/split w #"\,")]
      (if (< (count w1) (count w2))
        w2 w1))))


(defn get-file-name [file-path]
  (-> file-path
      (clojure.string/split #"/")
      (last)
      (clojure.string/split #"\.")
      (first)))


(defn get-file-ext [file-name]
  (-> file-name
      (clojure.string/split #"/")
      (last)
      (clojure.string/split #"\.")
      (last)
      (clojure.string/lower-case)))


(defmulti gen-mapping-by-file (fn [f-name _] (get-file-ext f-name)))


(defmethod gen-mapping-by-file :default
  [file-name file]
  (throw (emsg/ex-info-for-invalid-data-type (get-file-ext file-name))))


(defmethod gen-mapping-by-file "csv"
  [file-name file]
  (let [file-name (get-file-name file-name)
        file-name (clojure.string/lower-case file-name)
        header-list (get-header file)
        mapping (into [] (comp (map (fn [h]
                                      [file-name " " (clojure.string/lower-case h) " \"\"" "\n"]
                                      ))
                               cat
                               ) header-list)
        mapping (into mapping [file-name " \"\" " "(value_format 'json')" " \n"])]
    (clojure.string/join "" mapping)))


(defn format-for-web [mapping]
  (->> mapping
       (into [] (map (fn [v]
                       (str "\"" v "\"")
                       )))
       (partition 3)
       (reduce (fn [acc v]
                 (str acc (clojure.string/join " " v) "\n")
                 ) "")
       (clojure.string/join)))


(defmethod gen-mapping-by-file "json"
  [file-name file]
  (let [file-name (get-file-name file-name)
        file-name (clojure.string/lower-case file-name)]
    (->> (mcrawler/map-from-json-data-file file-name file)
         (format-for-web))))



(defmethod gen-mapping-by-file "xml"
  [file-name file]
  (let [file-name (get-file-name file-name)
        file-name (clojure.string/lower-case file-name)]
    (->> (mcrawler/from-xml-to-mapping-impl file-name file false)
         (format-for-web))))



(defn entries [zipfile]
  (enumeration-seq (.entries zipfile)))


(defmethod gen-mapping-by-file "zip"
  [file-name file]
  (with-open [z (java.util.zip.ZipFile. file)]
    (reduce (fn [acc ^java.util.zip.ZipEntry v]
              ;  (println (slurp (.getInputStream z v) ))
              ;  (println "--" (.getName v))
              (let [ext (get-file-ext (.getName v))
                    f-name (get-file-name (.getName v))
                    out (condp = ext
                          "xml"
                          (let [v (r/convert-xml-file-to-edn (.getInputStream z v) ) #_(->> (.getInputStream z v)
                                                                                                 xml/parse
                                                                                                 (xmltojson/parse)
                                                                                                 (xmltojson/format-map))]
                            (->> (mcrawler/from-edn-to-mapping f-name v )
                                 (format-for-web))
                            )

                          "json"
                          (gen-mapping-by-file (.getName v) (.getInputStream z v))
                          "csv"
                          (gen-mapping-by-file (.getName v) (.getInputStream z v))
                          (throw (emsg/ex-info-for-invalid-data-type (get-file-ext (.getName v)))))]
                (str acc "\n" out))
              ) "" (entries z))))



(defn gen-napping [topic-name edn-data-coll]
  (let [first-event (first edn-data-coll)
        first-event (w/stringify-keys first-event)
        first-event-str (pr-str first-event)]
    [topic-name "_" (str "(from_edn_data " first-event-str "  )")
     topic-name "\"\"" "(value_format 'json')"]))



(defmulti gen-mapping-by-file-name (fn [_ file-path]
                                     ; (println "--" file-path)
                                     (let [w (last (clojure.string/split file-path #"\."))]
                                       (clojure.string/lower-case w))))


(defmethod gen-mapping-by-file-name :default
  [topic-name file-path]
  (let [w (last (clojure.string/split file-path #"\."))]
    (throw (emsg/ex-info-for-invalid-data-type w) #_(ex-info "now support only json, xml and csv file " {:provided w
                                                                                                         :expected #{"json" "xml" "csv"}}))))


#_(defmulti gen-mapping-by-file-name (fn [_ file-path]
                                       ;  (println "--" (last (clojure.string/split file-path #"\.")))
                                       (let [w (last (clojure.string/split file-path #"\."))]
                                         (if (contains? #{"json" "xml" "csv"} w)
                                           w
                                           (throw (Exception. "now support only json, xml and csv file " {:provided w
                                                                                                          :expected #{"json" "xml" "csv"}
                                                                                                          }))))))


(defmethod gen-mapping-by-file-name "json"
  [topic-name json-file-path]
  (let []
    [topic-name "_" (str "(from_json_file \"" json-file-path "\" )")
     topic-name "_" "(value_format 'json' )" ";"]
    )
  )


(defmethod gen-mapping-by-file-name "csv"
  [topic-name csv-file-path]
  (let []
    [topic-name "_" (str "(from_csv_file \"" csv-file-path "\" )")
     topic-name "_" "(value_format 'json' )" ";"]
    )
  )


(defmethod gen-mapping-by-file-name "xml"
  [topic-name xml-file-path]
  (let []
    [topic-name "_" (str "(from_xml_file \"" xml-file-path "\" )")
     topic-name "_" "(value_format 'json' )" ";"]
    )
  )


(comment

  (gen-mapping-by-file-name "apl" "app/apl/apl_orginal_format.xml")

  )








(comment

  (gen-mapping-by-file "person.zip" "/home/mamun/person.zip")

  )