(ns ksql.gen.core_schema
  (:require [ksql.gen.file-util :as u]
            [ksql.gen.util :as cu]
            [ksql.gen.core-error-msg :as emsg]
            [ksql.gen.protocol :as p]
            [ingestion.api.client.ns-repo-impl :as c]
            ))


(defn get-source-schema [md-repo source-name]
  (reduce (fn [acc schema]
            (if (= (get schema :name) source-name)
              (reduced schema)
              acc
              )
            ) nil md-repo))


(defn get-fields [md-repo source-name]
  (reduce (fn [acc schema]
            (if (= (get schema :name) source-name)
              (let [fields (get schema :fields [])]
                (reduced fields))
              acc)
            ) nil md-repo))


(defn get-object-from-mapping [mapping]
  (into [] (rest (get (first mapping) :transfer_fn [[]] ))))


(defn get-subject-from-mapping [mapping]
  (get-in mapping [0 :name])
  #_(into [] (rest (get (first mapping) :transfer_fn [[]] ))))


(defn get-all-names [md-repo]
  (into [] (map :name) md-repo))


(defn get-property-names [md-repo source-name]
  (if-let [w (get-source-schema md-repo source-name)]
    (into [] (map :name ) (get w :fields))
    []))


(defn get-type [schema]
  (when-let [w (get schema :type)]
    ;    (first w)
    w
    ))


(defn get-key [schema]
  (when-let [w (get schema :key)]
    (first w)))


(defn get-value-format [schema]
  (when-let [w (get schema :value_format)]
    (first w)))


(defn gen-type? [m]
  (contains? #{"stream" "table" "mapping"} (get-type m)))


(defn as-vector [trans-fn]
  (clojure.walk/postwalk (fn [v]
                           (if (list? v)
                             (into [] v)
                             v
                             )) trans-fn))


(defn convert-fields-schema-from-mapping [md-repo mapping]
  (let [xf (comp
             (remove (fn [v] (nil? (:field_name v))))
             (map (fn [m]
                    ;(println "--" m )
                    (if (list? (:transfer_fn m))
                      (-> m
                          (clojure.set/rename-keys {:field_name :name})
                          (dissoc :topic :stream-type :partition)
                          (update :transfer_fn (fn [v] (into [] v)))
                          )
                      (-> m
                          (clojure.set/rename-keys {:field_name :name})
                          (dissoc :topic :stream-type :partition)
                          )
                      )
                    )))]
    (into [] xf mapping)))



(defn convert-where-schema-from-mapping [mapping]
  (let [xf (comp
             (filter (fn [v]
                       (and
                         (nil? (:field_name v))
                         (contains? #{"where"} (clojure.string/lower-case
                                                 (or (first (get v :transfer_fn [""])) "")
                                                 )))))
             (map :transfer_fn)
             (map rest)
             cat
             )]

    (into [] xf mapping)
    )
  )


(defn convert-join-schema-from-mapping [mapping]
  (let [xf (comp
             (filter (fn [v]
                       (and
                         (nil? (:field_name v))
                         (contains? #{"join" "left_join" "outer_join"} (first (get v :transfer_fn [""]))))))
             (map :transfer_fn)
             )

        ]
    (into [] xf mapping)
    )
  )



(defn get-group-by-from-mapping [mapping]
  (let [xf (comp
             (filter (fn [v]
                       (and
                         (nil? (:field_name v))
                         (contains? #{"group_by"} (first (get v :transfer_fn [""]))))))
             (map :transfer_fn)
             (map rest #_(juxt first rest))
             cat
             )

        ; _ (println "----" group-by2)
        ]
    (into [] xf mapping)

    )
  )


(defn get-validation-from-mapping [mapping]
  (let [xf (comp
             (filter (fn [v]
                       (and
                         (nil? (:field_name v))
                         (contains? #{"validation!"} (first (get v :transfer_fn [""]))))))
             (map :transfer_fn)
             (map rest)
             cat
             cat
             )

        ]
    (into [] xf mapping)

    )
  )


(defn get-type-from-mapping [mapping]
  (let [xf (comp
             (filter (fn [v]
                       (and
                         (nil? (:field_name v))
                         (contains? #{"type"} (first (get v :transfer_fn [""]))))))
             (map :transfer_fn)
             (map rest)
             cat

             )

        ]
    (into [] xf mapping)

    )
  )



(defn convert-mapping-to-md-schema [md-repo mapping]
  ; (p/log-v mapping)
  (let [source-name (cu/get-source-name2 mapping)
        sink-name (get-in mapping [0 :name])

        all-names (into #{} (get-all-names md-repo))


        {:keys [topic partition stream-type]} (first mapping)


        ;kafka_topic='error', value_format='AVRO'



        type (when-let [t (get-type-from-mapping mapping)]
               (first t))

        type (if (contains? all-names sink-name)
               "mapping"
               (or type stream-type "stream")
               )

        fields    (convert-fields-schema-from-mapping md-repo mapping)
        join      (convert-join-schema-from-mapping mapping)
        where     (convert-where-schema-from-mapping mapping)
        group-by2 (get-group-by-from-mapping mapping)
        validateion! (get-validation-from-mapping mapping)


     ;   _ (p/log-v mapping)
        schema-m (->> mapping
                      (filter (fn [v] (nil? (:field_name v))))
                      (remove (fn [v]
                                (when (nil? (get v :transfer_fn ))
                                  (throw (emsg/ex-info-for-null-field-or-transformation v) #_(ex-info "invalid field or transformation " {
                                                                           :e-ref   v #_(clojure.set/difference mapping-fields sink-fields)

                                                                           :e-des   "invalid field or transformation "
                                                                           }))
                                  )
                                (contains? #{"join" "left_join" "outer_join" "where" "group_by" "validation!"} (clojure.string/lower-case (first (get v :transfer_fn [""]))))))
                      (mapv :transfer_fn)
                      (group-by first)
                      (mapv (fn [[k v]]
                              (let [k1 (keyword k)
                                    v2 (into [] (comp (map rest) cat) v)]
                                {k1 v2})))
                      (into {}))

        value_format (or (first (get schema-m :value_format []))
                         "AVRO")

        value_format (if (clojure.string/starts-with? value_format "'")
                       value_format
                       (str "'" value_format "'"))

        partitions (or (first (get schema-m :partitions []))
                       (get p/context :partitions))

        offset (or (first (get schema-m :offset []))
                   (get p/context :offset))

        offset (get #{"earliest" "latest"} offset "latest")

        with (if (or (= type "stream")
                     (= type "table"))
               {:kafka_topic  (or topic sink-name)
                :value_format  value_format
                :partitions partitions
                :replicas (get p/context :replicas)
                :offset offset}
               {}
               )

        schema-m (-> (assoc schema-m :with with)
                     (dissoc :partitions :replicas :value_format :join :left_join :outer_join))]


    (cond-> (merge schema-m
                   {:name sink-name :type type})

            (and source-name
                 (not (empty? source-name)))
            (assoc :source-name source-name)

            (and fields
                 (not (empty? fields)))
            (assoc :fields fields)

            (and join
                 (not (empty? join)))
            (assoc :join join)

            (and where
                 (not (empty? where)))
            (assoc :where where)

            (and group-by2
                 (not (empty? group-by2)))
            (assoc :group_by group-by2)

            (and validateion!
                 (not (empty? validateion!)))
            (assoc :validation! validateion!)

            partition (assoc :partitions partition)
            topic (assoc :topic topic))))

(comment

  (->> [[:group_by :person/id :person/name]]
       (into {} (map (juxt first rest)))
       )
  )

