(ns day8.re-frame-10x.view.components
  (:require [clojure.string :as str]
            [goog.fx.dom :as fx]
            [day8.re-frame-10x.inlined-deps.re-frame.v0v11v0.re-frame.core :as rf]
            [clojure.string :as str]
            [day8.re-frame-10x.utils.re-com :as rc]
            [day8.re-frame-10x.inlined-deps.reagent.v0v9v1.reagent.core :as r]
            [devtools.prefs]
            [devtools.formatters.core]
            [day8.re-frame-10x.svgs :as svgs]
            ["react-highlight.js" :as react-highlightjs]
            ["highlight.js/lib/languages/clojure"])
  (:require-macros [day8.re-frame-10x.utils.macros :refer [with-cljs-devtools-prefs]]))

(defn search-input [{:keys [title placeholder on-save on-change on-stop]}]
  (let [val  (r/atom title)
        save #(let [v (-> @val str str/trim)]
                (when (pos? (count v))
                  (on-save v)))]
    (fn []
      [:input {:type        "text"
               :value       @val
               :auto-focus  true
               :placeholder placeholder
               :size        (if (> 20 (count (str @val)))
                              25
                              (count (str @val)))
               :on-change   #(do (reset! val (-> % .-target .-value))
                                 (on-change %))
               :on-key-down #(case (.-which %)
                               13 (do
                                    (save)
                                    (reset! val ""))
                               nil)}])))

(defn scroll! [el start end time]
  (.play (fx/Scroll. el (clj->js start) (clj->js end) time)))

(defn scrolled-to-end? [el tolerance]
  ;; at-end?: element.scrollHeight - element.scrollTop === element.clientHeight
  (> tolerance (- (.-scrollHeight el) (.-scrollTop el) (.-clientHeight el))))

;; Data browser

(defn string->css [css-string]
  "This function converts jsonml css-strings to valid css maps for hiccup.
  Example: 'margin-left:0px;min-height:14px;' converts to
           {:margin-left '0px', :min-height '14px'}"

  (->> (str/split css-string #";")
       (map #(str/split % #":"))
       (reduce (fn [acc [property value]]
                 (assoc acc (keyword property) value)) {})))

(declare jsonml->hiccup)

(def default-cljs-devtools-prefs @devtools.prefs/default-config)

(defn reset-wrapping [css-string]
  (str/replace css-string #"white-space:nowrap;" ""))

(def customized-cljs-devtools-prefs
  {; Override some cljs-devtools default styles.

   ; The goal here is to make default styles more flexible and wrap at the edge of our panel (we don't want horizontal
   ; scrolling). Technically we want to remove all 'white-space:no-wrap'.
   ; See https://github.com/binaryage/cljs-devtools/blob/master/src/lib/devtools/defaults.cljs
   ;; Commented out as this causes some other issues too.
   ;:header-style (reset-wrapping (:header-style default-cljs-devtools-prefs))
   ;:expandable-style (reset-wrapping (:expandable-style default-cljs-devtools-prefs))
   ;:item-style (reset-wrapping (:item-style default-cljs-devtools-prefs))

   ; Hide the index spans on the left hand of collections. Shows how many elements in a collection.
   :none-style                     "display: none"
   :index-tag                      [:span :none-style]
   :min-expandable-sequable-count-for-well-known-types
                                   3

   ; Our JSON renderer does not have hierarchy depth limit,
   ; See https://github.com/binaryage/cljs-devtools/blob/master/src/lib/devtools/formatters/budgeting.cljs
   :initial-hierarchy-depth-budget false})

(def effective-cljs-devtools-prefs (merge default-cljs-devtools-prefs customized-cljs-devtools-prefs))

(defn make-devtools-api-call [api-fn & args]
  (with-cljs-devtools-prefs effective-cljs-devtools-prefs
                            (apply api-fn args)))

(defn cljs-devtools-header [& args]
  (apply make-devtools-api-call devtools.formatters.core/header-api-call args))

(defn cljs-devtools-body [& args]
  (apply make-devtools-api-call devtools.formatters.core/body-api-call args))

(defn cljs-devtools-has-body [& args]
  (apply make-devtools-api-call devtools.formatters.core/has-body-api-call args))

(defn get-object [jsonml]
  (.-object (get jsonml 1)))

(defn get-config [jsonml]
  (.-config (get jsonml 1)))

(defn data-structure [jsonml path]
  (let [expanded? (rf/subscribe [:app-db/node-expanded? path])]
    (fn [jsonml path]
      [:span
       {:class (str "re-frame-10x--object" (when @expanded? " expanded"))}
       [:span {:class    "toggle"
               :on-click #(rf/dispatch [:app-db/toggle-expansion path])}
        [:button.expansion-button (if @expanded? [svgs/up-arrow :fill "#6EC0E6"] [svgs/down-arrow :fill "#6EC0E6"])]]
       (if (and @expanded? (cljs-devtools-has-body (get-object jsonml) (get-config jsonml)))
         (jsonml->hiccup
           (cljs-devtools-body
             (get-object jsonml)
             (get-config jsonml))
           (conj path :body))
         (jsonml->hiccup
           (cljs-devtools-header
             (get-object jsonml)
             (get-config jsonml))
           (conj path :header)))])))

(defn jsonml->hiccup
  "JSONML is the format used by Chrome's Custom Object Formatters.
  The spec is at https://docs.google.com/document/d/1FTascZXT9cxfetuPRT2eXPQKXui4nWFivUnS_335T3U/preview.

  JSONML is pretty much Hiccup over JSON. Chrome's implementation of this can
  be found at https://cs.chromium.org/chromium/src/third_party/WebKit/Source/devtools/front_end/object_ui/CustomPreviewComponent.js
  "
  [jsonml path]
  (if (number? jsonml)
    jsonml
    (let [[tag-name attributes & children] jsonml
          tagnames #{"div" "span" "ol" "li" "table" "tr" "td"}]
      (cond
        (contains? tagnames tag-name) (into
                                        [(keyword tag-name) {:style (-> (js->clj attributes)
                                                                        (get "style")
                                                                        (string->css))}]
                                        (map-indexed (fn [i child] (jsonml->hiccup child (conj path i))))
                                        children)

        (= tag-name "object") [data-structure jsonml path]
        :else jsonml))))

(defn subtree [data title path]
  (let [expanded? (rf/subscribe [:app-db/node-expanded? path])]
    (fn [data]
      [rc/v-box
       :children
       [[rc/h-box
         :align :center
         :class (str/join " " ["re-frame-10x--object"
                               (when @expanded? "expanded")])
         :children
         [[:span {:class    "toggle"
                  :on-click #(rf/dispatch [:app-db/toggle-expansion path])}
           [:button.expansion-button (if @expanded? "▼ " "▶ ")]]
          (or title "data")]]
        [rc/h-box
         :children [[:div {:style {:margin-left 20}}
                     (cond
                       (and @expanded?
                            (or (string? data)
                                (number? data)
                                (boolean? data)
                                (nil? data))) [:div {:style {:margin "10px 0"}} (prn-str data)]
                       @expanded? (jsonml->hiccup (cljs-devtools-header data) (conj path 0)))]]]]])))

(defn subscription-render [data title path]
  (let [expanded? (r/atom true) #_(rf/subscribe [:app-db/node-expanded? path])]
    (fn [data]
      [:div
       {:class (str/join " " ["re-frame-10x--object"
                              (when @expanded? "expanded")])}
       #_[:span {:class    "toggle"
                 :on-click #(rf/dispatch [:app-db/toggle-expansion path])}
          [:button.expansion-button (if @expanded? "▼ " "▶ ")]]
       (or title "data")
       [:div {:style {:margin-left 20}}
        (cond
          (and @expanded?
               (or (string? data)
                   (number? data)
                   (boolean? data)
                   (nil? data))) [:div {:style {:margin "10px 0"}} (prn-str data)]
          @expanded? (jsonml->hiccup (cljs-devtools-header data) (conj path 0)))]])))

(defn simple-render [data path & [class]]
  (let [expanded? (r/atom true) #_(rf/subscribe [:app-db/node-expanded? path])]
    (fn [data]
      [:div
       {:class (str/join " " ["re-frame-10x--object"
                              (when @expanded? "expanded")
                              class])}
       #_[:span {:class    "toggle"
                 :on-click #(rf/dispatch [:app-db/toggle-expansion path])}
          [:button.expansion-button (if @expanded? "▼ " "▶ ")]]
       [:div #_{:style {:margin-left 20}}
        (cond
          (and @expanded?
               (or (string? data)
                   (instance? js/RegExp data)
                   (number? data)
                   (boolean? data)
                   (nil? data))) [:div {:style {:margin "10px 0"}} (prn-str data)]
          @expanded? (jsonml->hiccup (cljs-devtools-header data) (conj path 0)))]])))

(defn tag [class label]
  [rc/box
   :class (str "rft-tag noselect " class)
   :child [:span {:style {:margin "auto"}} label]])

;; shadow-cljs uses the react-highlight NPM package, whereas figwheel etc use a CLJSJS package. The CLJSJS package
;; unfortunately incorrectly 'unwraps' the default export making the two incompatible. So we must check if we can
;; use the default export first, otherwise use the ns directly.
;; See: https://github.com/cljsjs/packages/blob/master/react-highlight/resources/main.js
(def highlight (r/adapt-react-class (or react-highlightjs/default react-highlightjs)))
