(ns active.clojure.config.io
  (:require [active.clojure.config :as config]
            [active.clojure.logger.event :as log]))

(defn try-read-config-file
  [filename schema & profiles]
  (log/log-event! :info (log/log-msg "Trying configuration file:" filename))
  (try (let [config (config/make-configuration schema profiles
                                               (read-string (slurp filename)))]
         (log/log-event! :info (log/log-msg "Using configuration file:" filename))
         config)
       (catch java.io.FileNotFoundException _e
         (log/log-event! :warn (log/log-msg "Configuration file not found:" filename))
         nil)
       (catch java.lang.IllegalArgumentException e
         (log/log-exception-event! :error (log/log-msg "Error parsing configuration file: " filename (.getMessage e)) e)
         nil)
       (catch Exception e
         (log/log-exception-event! :error (log/log-msg "Error reading configuration file:" filename (.getMessage e)) e))))

(defn try-save-config-file
  [filename config]
  (log/log-event! :debug (log/log-msg "Saving configuration file:" filename))
  (try (let [edn-config (config/configuration-object config)]
         (spit filename edn-config))
       (catch Exception e
         (log/log-exception-event! :error (log/log-msg "Error writing configuration file:" (.getMessage e)) e))))
