(ns ^{:doc "A small layer over the primitive dom elements of [[reacl2.dom]], where all event-handlers must return a [[reacl2.core/return]] value instead.

  If no event-handlers are registered, then the functions here return a
  primitive element identical to their counterparts
  in [[reacl2.dom]]. If event handlers are used, then a class instance
  is returned instread.

  Note that the event-handler must not 'return' a new :app-state or :local-state; only :message and :action are meaningful.
"}  reacl-basics.adom
  (:require-macros [reacl-basics.adom :refer [defdom]])
  (:require [react :as react]
            [reacl2.core :as reacl :include-macros true]
            [reacl2.dom :as dom])
  (:refer-clojure :exclude (meta map time use set symbol)))

(defn- is-event? [k]
  (.startsWith (name k) "on"))

(defn- raw-event-handler [comp event f]
  (fn [ev]
    (let [a (f ev)]
      (when (some? a)
        (when-not (reacl/returned? a)
          (throw (ex-info (str "Expected a 'return' value to be returned from " event ", but got " (pr-str a)) {:value a :event event})))
        (when-not (reacl/keep-state? (reacl/returned-app-state a))
          (throw (ex-info (str "An event handler must not return a app-state update, but one was returned from " event) {:event event})))
        (when-not (reacl/keep-state? (reacl/returned-local-state a))
          (throw (ex-info (str "An event handler must not return a local-state update, but one was returned from " event) {:event event})))
        (reacl/send-message! comp a)))))

(defn- event-handler [^js/WeakMap cache comp event f]
  (if (.has cache f)
    (.get cache f)
    (let [h (raw-event-handler comp event f)]
      (.set cache f h)
      h)))

(defn ^:no-doc adom-function [n]
  (let [elem (partial dom/element n)

        class (reacl/class (name (clojure.core/symbol (str *ns*) n)) this [attrs & children]
                           local-state [cache (js/WeakMap.)]
                           
                           render
                           (apply elem (into {} (clojure.core/map (fn [[k v]]
                                                                    [k (if (is-event? k)
                                                                         (event-handler cache this k v)
                                                                         v)])
                                                                  attrs))
                                  children)
                           
                           handle-message identity)]
    (fn [& args]
      (let [[attrs children] (if (and (not (empty? args)) (dom/attributes? (first args)))
                               [(first args) (rest args)]
                               [nil args])]
        (if (some is-event? (keys attrs))
          (apply class attrs children)
          (apply elem attrs children))))))

;; The following HTML elements are supported by react (http://facebook.github.io/react/docs/tags-and-attributes.html)
(defdom a)
(defdom abbr)
(defdom address)
(defdom area)
(defdom article)
(defdom aside)
(defdom audio)
(defdom b)
(defdom base)
(defdom bdi)
(defdom bdo)
(defdom big)
(defdom blockquote)
(defdom body)
(defdom br)
(defdom button)
(defdom canvas)
(defdom caption)
(defdom cite)
(defdom code)
(defdom col)
(defdom colgroup)
(defdom data)
(defdom datalist)
(defdom dd)
(defdom del)
(defdom details)
(defdom dfn)
(defdom div)
(defdom dl)
(defdom dt)
(defdom em)
(defdom embed)
(defdom fieldset)
(defdom figcaption)
(defdom figure)
(defdom footer)
(defdom form)
(defdom h1)
(defdom h2)
(defdom h3)
(defdom h4)
(defdom h5)
(defdom h6)
(defdom head)
(defdom header)
(defdom hr)
(defdom html)
(defdom i)
(defdom iframe)
(defdom img)
(defdom input)
(defdom ins)
(defdom kbd)
(defdom keygen)
(defdom label)
(defdom legend)
(defdom li)
(defdom link)
(defdom main)
(defdom map)
(defdom mark)
(defdom menu)
(defdom menuitem)
(defdom meta)
(defdom meter)
(defdom nav)
(defdom noscript)
(defdom object)
(defdom ol)
(defdom optgroup)
(defdom option)
(defdom output)
(defdom p)
(defdom param)
(defdom pre)
(defdom progress)
(defdom q)
(defdom rp)
(defdom rt)
(defdom ruby)
(defdom s)
(defdom samp)
(defdom script)
(defdom section)
(defdom select)
(defdom small)
(defdom source)
(defdom span)
(defdom strong)
(defdom style)
(defdom sub)
(defdom summary)
(defdom sup)
(defdom table)
(defdom tbody)
(defdom td)
(defdom textarea)
(defdom tfoot)
(defdom th)
(defdom thead)
(defdom time)
(defdom title)
(defdom tr)
(defdom track)
(defdom u)
(defdom ul)
(defdom var)
(defdom video)
(defdom wbr)
(defdom svg)
(defdom polygon)
(defdom line)
(defdom rect)
(defdom circle)
(defdom ellipse)
(defdom polyline)
(defdom text)
(defdom path)
(defdom defs)
(defdom clipPath)
(defdom g)
(defdom linearGradient)
(defdom radialGradient)
(defdom stop)
(defdom image)
(defdom animate)
(defdom animateColor)
(defdom animateMotion)
(defdom animateTransform)
(defdom set)
(defdom cursor)
(defdom desc)
(defdom feBlend)
(defdom feColorMatrix)
(defdom feComponentTransfer)
(defdom feComposite)
(defdom feConvolveMatrix)
(defdom feDiffuseLighting)
(defdom feDisplacementMap)
(defdom feDistantLight)
(defdom feFlood)
(defdom feFuncA)
(defdom feFuncB)
(defdom feFuncG)
(defdom feFuncR)
(defdom feGaussianBlur)
(defdom feImage)
(defdom feMerge)
(defdom feMergeNode)
(defdom feMorphology)
(defdom feOffset)
(defdom fePointLight)
(defdom feSpecularLighting)
(defdom feSpotLight)
(defdom feTile)
(defdom feTurbulence)
(defdom font)
(defdom marker)
(defdom mask)
(defdom metadata)
(defdom mpath)
(defdom pattern)
(defdom switch)
(defdom symbol)
(defdom textPath)
(defdom tspan)
(defdom use)
(defdom view)

