(ns sqlosure.interpreter.sexp
  (:require [active.clojure.condition :as c]
            [clojure.set :as set]
            [sqlosure.relational-algebra :as rel]))

(defn alist-lookup-value
  "Return all keys to a value from an alist."
  [alist val]
  (reduce (fn [acc [k v]]
            (if (= val v)
              (conj acc k)
              acc))
          []
          alist))

(defn extend-keys
  [m extensions]
  (reduce (fn [acc [k v]]
            (let [ex (alist-lookup-value extensions k)]
              (if (empty? ex)
                (assoc acc k v)
                (merge acc (into {} (map (fn [e] [e v]) ex))))))
          {}
          m))

(defn rand->value
  [row r]
  (cond
    (rel/const? r)
    (rel/const-val r)

    (rel/attribute-ref? r)
    (get row (rel/attribute-ref-name r))))

(defn- some-aggregation?
  [thing]
  (or (rel/aggregation? thing)
      (rel/aggregation-all? thing)))

(defn unroll-project-alist
  "Takes a projection's alist and returns an unrolled alist with aggregations
  wrapped in a record for further interpretation."
  [alist]
  (mapv (fn [[k v]]
          [k (if (some-aggregation? v)
               v
               (rel/attribute-ref-name v))])
        alist))

(defn alist->aggregations
  "Does an `alist` contain an aggregation?"
  [alist]
  (filter (comp some-aggregation? second) alist))

(defn interpret-aggregation
  [[field aggr] rows]
  (cond
    (rel/aggregation? aggr)
    (let [attr (rel/attribute-ref-name (rel/aggregation-expr aggr))
          rows (filter #(get % attr) rows)]
      {field (case (rel/aggregation-operator aggr)
               :count (count (filter #(get % attr) rows))
               :sum   (reduce + 0 (map #(get % attr) rows))
               :avg   (/ (reduce + 0 (map #(get % attr) rows))
                         (count rows))
               :min   (apply min (map #(get % attr) rows))
               :max   (apply max (map #(get % attr) rows))
               (:std-dev :std-dev-p :var :var-p)
               (throw (java.lang.UnsupportedOperationException. "no implemented yet")))})

    (rel/aggregation-all? aggr)
    {field (case (rel/aggregation-all-operator aggr)
             :count-all (count rows)

             (c/assertion-violation `interpret-aggregation "not a valid aggregation" aggr))}))

(defn interpret
  [db q]
  (cond
    (rel/empty-query? q)
    nil

    (rel/base-relation? q)
    (get db (rel/base-relation-handle q))

    (rel/project? q)
    (let [sub-q        (rel/project-query q)
          alist        (rel/project-alist q)
          extensions   (unroll-project-alist alist)
          sub-res      (interpret db sub-q)
          aggregations (alist->aggregations extensions)]
      ;; We need to look ahead for grouping sub-queries because
      ;; there are some restrictions on how those need to be evaluated.
      ;; Especially, a grouping query must be projected via an aggregation.
      (cond
        ;; Base case
        (and (empty? aggregations)
             (not (rel/group? sub-q)))
        (let [select (mapv (comp rel/attribute-ref-name second) alist)]
          (mapv (fn [row]
                  (println "row" row "select" select "extend" extensions)
                  (extend-keys (select-keys row select) extensions))
                sub-res))

        (and aggregations
             (not (rel/group? sub-q)))
        (let [aggr (first aggregations)]
          [(interpret-aggregation aggr sub-res)])

        (and aggregations
             (rel/group? sub-q))
        (let [groups sub-res
              aggr (first aggregations)
              projections (filter (comp (comp not some-aggregation?) second) extensions)
              select (mapv (comp rel/attribute-ref-name second) projections)]
          (map (fn [[grp rows]]
                 (let [aggr-rows (interpret-aggregation aggr rows)
                       non-aggr-rows (map (fn [row]
                                            (extend-keys (select-keys row select) projections))
                                          rows)]
                   (map merge aggr-rows non-aggr-rows)))
               groups))))

    (rel/restrict? q)
    (let [app   (rel/restrict-exp q)
          query (rel/restrict-query q)
          rator (rel/application-rator app)
          ;; each rator is either a constant or an attribute ref
          rands #(map (partial rand->value %) (rel/application-rands app))
          sub   (interpret db query)]

      (into [] (filter (fn [row]
                         (apply (rel/rator-proc rator) (rands row))) sub)))

    (rel/restrict-outer? q)
    nil

    (rel/combine? q)
    nil

    (rel/order? q)
    (let [[ref direction] (first (rel/order-alist q))
          attr            (rel/attribute-ref-name ref)
          sub             (interpret db (rel/order-query q))
          res             (sort-by #(get % attr) sub)]
      (if (= :ascending direction)
        res
        (reverse res)))

    (rel/group? q)
    (let [cols (rel/group-columns q)
          sub  (interpret db (rel/group-query q))]
      (group-by #(get % (first cols)) sub))

    (rel/top? q)
    (let [cnt    (rel/top-count q)
          offset (rel/top-offset q)
          sub    (interpret db (rel/top-query q))]
      (take cnt (drop (or offset 0) sub)))

    (rel/distinct-q? q)
    (let [sub (interpret db (rel/distinct-q-query q))]
      (distinct sub))

    :else
    (c/assertion-violation `interpret "not a query" q)))
