(ns forms-bootstrap.core
  (:use net.cgrand.enlive-html
        noir.core
        forms-bootstrap.util
        [validateur.validation :only (valid?)])
  (:require [noir.validation :as vali]
            [clojure.string :as string]
            [noir.session :as session]
            [noir.response :as response]))

(def form-template "forms_bootstrap/forms-template.html")
;;HELPER FUNCTIONS

(defn handle-error-css
  "Used within a snippet to add 'error' to the class of an html element, if appropriate."
  [errors]
  (if (peek errors)
     (add-class "error")
     identity))

(defn span
  "Creates an enlive node map for an html span element. Type should be
  'help-inline' or 'help-block.'"
  [type content]
  {:tag "span"
   :attrs {:class type}
   :content content})

(defn add-spans
  "Used by enlive snippets to add an inline span to display errors (if
  there are any). If there are no errors then it adds any inline or
  block help messages."
  [errors help-inline help-block]
  (if (peek errors)
    (append (span "help-inline" (peek errors)))
    (do->
     (if (seq help-inline) 
       (append (span "help-inline" help-inline))
       identity)
     (if (seq help-block)
       (append (span "help-block" help-block))
       identity))))
  

;; SNIPPETS  

;;Grabs the whole form from the template, sets the action, replaces
;;the entire contents with fields, and appends a submit button.
;;Fields here is a sequence of stuff generated by all of the
;;defsnippets below, one for each form element
(defsnippet basic-form  
  form-template        
  [:form]              
  [{:keys [action fields submitter class enctype legend]
    :or {class "form-horizontal"}}]
  [:form] (do-> (set-attr :action action
                          :class class)
                (if (seq enctype)
                  (set-attr :enctype enctype)
                  identity)
                (content (if (empty? legend)
                           fields
                           (list {:tag "legend" 
                                  :content legend}
                                 fields)))
                (append submitter)))

;; Generates a txt or password input form field for a given map.
;; inputs: name label type size default errors
;; ex: [:name "namehere" :label "labelhere" :type "text"
;;      :size "xlarge" :errors ["some error"] :default "john"]
(defsnippet input-lite
  form-template
  [:div.input-field :input]
  [{:keys [id hidden name label class type size errors default
           disabled placeholder value onclick] 
    :or {size "input-large"         
         default ""}}]
  [:input] (do->
            (set-attr :name name)            
            (set-attr :type type)
            (set-attr :class size)
            (set-attr :placeholder placeholder)
            (if (= type "button")
              (set-attr :value value)
              (set-attr :value default))            
            (if (seq onclick)
              (set-attr :onclick onclick)
              identity)
            (if (= disabled true)
              (set-attr :disabled "")
              identity)
            (add-class class)))

(defsnippet input-field
  form-template
  [:div.input-field]
  [{:keys [id hidden name label class type size errors default disabled
           placeholder help-inline help-block] 
    :as m}]
  [:.input-field] (do->
                   (if id
                     (set-attr :id id)
                     identity)
                   (if hidden 
                     (add-class "hidden")
                     identity)
                   (handle-error-css errors))    
  [:label] (do-> (content label) 
                 (set-attr :for name)) 
  [:div.controls :input] (constantly (input-lite m))
  [:div.controls]   (add-spans errors help-inline help-block))
 
;;Creates a text-area form element
;;ex: [:name "namehere" :label "labelhere" :type "text-area"
;;     :size "xlarge" :rows "3" :default "defaultstuff"]
(defsnippet text-area-lite
  form-template
  [:div.text-area :textarea]
  [{:keys [name label size rows errors default class]
    :or {size "input-large"
         rows "3"
         default ""}}]
  [:textarea] (do-> (set-attr :class size)
                    (set-attr :name name)
                    (set-attr :rows rows)
                    (add-class class)
                    (content default)))

(defsnippet text-area-field  
  form-template
  [:div.text-area]
  [{:keys [name label size rows errors default help-inline help-block] :as m}]
  [:div.text-area] (handle-error-css errors)
  [:label] (do-> (set-attr :for name)
                 (content label)) 
  [:textarea] (constantly (text-area-lite m))
  [:div.controls] (add-spans errors help-inline help-block))

;;Creates a select (dropdown) form element with the given values 
;;Ex: {:type "select" :name "cars" :size "xlarge" :label "Cars"
;;     :inputs [["volvo" "Volvo"] ["honda" "Honda"]]}
(defsnippet select-lite
  form-template
  [:div.select-dropdown :select]
  [{:keys [name size class label inputs errors default type]
    :or {size "input-large"}}]
  [:select] (do->
             (set-attr :name name
                       :id name
                       :class size)
             (add-class class)
             (if (string-contains? type "multiple")
               (set-attr :multiple "multiple")
               identity))
  [:option] (clone-for [[value value-label] inputs]
                       (do-> (set-attr :value value)
                             (if (= default value)
                               (set-attr :selected "selected")  
                               identity)
                             (content value-label))))

(defsnippet select-field
  form-template                       
  [:div.select-dropdown] 
  [{:keys [name size label inputs errors default type help-inline help-block]
    :as m}]
  [:div.select-dropdown] (handle-error-css errors)
  [:label] (do-> (content label) 
                 (set-attr :for name))
  [:select] (constantly (select-lite m))
  [:div.controls]  (add-spans errors help-inline help-block))
 
;;Creates a radio or checkbox form list with the given attributes
;; ex: {:type "select" :name "cars" :size "xlarge" :label "Cars"
;;      :inputs [["volvo" "Volvo"] ["honda" "Honda"]]}
(defsnippet checkbox-or-radio-lite
  form-template 
  [:div.checkbox-or-radio :div.controls :label]
  [{:keys [name inputs type errors default]}]
  [:label] (clone-for [[value value-label] inputs] 
                      [:label] (do-> (add-class type)
                                     (if (string-contains? type "inline")
                                       (add-class "inline")
                                       identity))
                      [:input] (do-> (set-attr :type (first-word type)
                                               :name name
                                               :value value
                                               :id (remove-spaces
                                                    value-label))
                                     (content value-label)
                                     (if (contains?
                                          (set (collectify default))
                                          value)
                                       (set-attr :checked "checked")
                                       identity))))

(defsnippet checkbox-or-radio 
  form-template                       
  [:.checkbox-or-radio] 
  [{:keys [class name label inputs errors help-inline help-block] :as m}]
  [:div.checkbox-or-radio]  (add-class class) ;;'checkbox' or 'radio'
  [:div.checkbox-or-radio] (handle-error-css errors)
  [:label.control-label] (do-> (content label)  
                               (set-attr :name name))
  [:div.controls :label] (content
                          (checkbox-or-radio-lite m))
  [:div.controls] (add-spans errors help-inline help-block))
   
;;Creates a file input button
(defsnippet file-input-lite
  form-template
  [:div.file-input :input]
  [{:keys [name label errors]}]
  [:input] (set-attr :name name))

(defsnippet file-input
  form-template
  [:div.file-input]
  [{:keys [name label errors help-inline help-block] :as m}]
  [:div.file-input] (handle-error-css errors)
  [:label] (content label)
  [:input] (constantly (file-input-lite m))
  [:div.controls] (add-spans errors help-inline help-block))

;;Creates a submit button with a specified label (value)
;;TODO: add support for different types of buttons within <div class="actions">
(defsnippet button-lite
  form-template
  [:div#submit-button :button]
  [label class] 
  [:button] (do-> (content label)
                  (add-class class)))

(defsnippet make-submit-button  
  form-template
  [:div#submit-button]
  [label cancel-link]  
  [:button] (constantly (button-lite label "btn-primary"))
  [:a] (if cancel-link
         (if (= cancel-link "modal") 
           (set-attr :data-dismiss "modal")
           (set-attr :href cancel-link))
         (content "")))

(defsnippet inline-fields
  form-template
  [:div.inline-fields]
  [{:keys [name label type columns]}]
  [:label] (content label)
  [:div.inline-fields] (handle-error-css (some (fn[a] a) (map :errors columns)) )
  [:div.controls-row] (content  (interpose " "
                                           (for [item columns]
                                             (if (string-contains? (:type item) "text")
                                               (input-lite item)
                                               (if (string-contains? (:type item) "select")
                                                 (select-lite item)
                                                 identity))))
                                (when-let [err-msg (some (fn[a] a) (map :errors columns))]
                                  {:tag "span"
                                   :attrs {:class "help-inline"}
                                   :content err-msg})))

;;HELPERS
(defn make-field-helper
  [form-class field field-lite m]
  (if (string-contains? form-class "form-inline")
    (list (field-lite m) " ")
    (field m)))

(defn make-field
  "Takes a single map representing a form element's attributes and
  routes it to the correct snippet based on its type. Supports input
  fields, text areas, dropdown menus, checkboxes, radios, and file
  inputs. Ex: {:type 'text' :name 'username' :label 'Username' :errors
  ['Incorrect username'] :default ''}"
  [form-class m]
  (case (first-word (:type m))
    "text"       (make-field-helper form-class
                                    input-field input-lite m)
    "password"   (input-field (dissoc m :default)) ;;dont keep on
    ;;render
    "button" (make-field-helper form-class
                                input-field input-lite m)
    "text-area"  (make-field-helper form-class 
                                    text-area-field text-area-lite m)
    "select"     (make-field-helper form-class
                                    select-field select-lite m)
    "radio"      (make-field-helper form-class 
                                    checkbox-or-radio checkbox-or-radio-lite m) 
    "checkbox"   (make-field-helper form-class
                                    checkbox-or-radio checkbox-or-radio-lite m)
    "inline-fields" (inline-fields m)
    "file-input" (file-input m)))

(defn inline-errs-defs
  "Used by make-form to add errors and defaults for form fields of
  type 'inline-fields.' Adds an :errors and :default to each inline
  field."
  [{:keys [columns] :as m} errors-and-defaults]
  (let [new-columns (vec
                     (map #(merge %1
                                  (get errors-and-defaults
                                       (keyword (:name %1))))
                          columns))]
    (merge m {:columns new-columns})))

;;  ex: ({:type "text" :name "username" :label "Username"}
;;       {:type "pass" :name "password" :label "Password"})
;; after we merge with errors / defs, one field could look like:
;; {:type "text" :name "username" :errors ["username cannot be blank"] :default ""}
;;Submit-Label is the label on the submit button - default is "Submit"
(defn make-form
  "Returns a form with the specified action, fields, and submit
  button. Fields is a sequence of maps, each containing a form element's
  attributes."
  [& {:keys [action class fields submit-label errors-and-defaults
             enctype cancel-link legend button-type] :as form-map
      :or {class "form-horizontal"}}]
;;    (println "make-form input map: " form-map "\n")
  (basic-form {:action action
               :legend legend
               :class class
               :enctype  enctype
               :fields   (map (fn [{:keys [name type] :as a-field}]
                                (make-field
                                 class
                                 (merge a-field
                                        (if (string-contains? type "inline-fields")
                                          (inline-errs-defs a-field errors-and-defaults)
                                          (get errors-and-defaults
                                               (keyword
                                                ;;replace [] in case its
                                                ;;the name of a form
                                                ;;element that can take
                                                ;;on mutliple values (ie checkbox)
                                                (string/replace name "[]" "")))))))
                              fields) 
               :submitter (if (string-contains? class "form-inline")
                            (button-lite submit-label button-type)
                            (when submit-label
                              (make-submit-button submit-label cancel-link)))}))


;;MACROS

;; @vali/*errors* looks like:
;;  {:title [["title must be an integer number!"]], :location
;;  [["location cannot be blank!"]]}
;; 'errors' are the sandbar :_validation-errors from the result of validating
;;  the form map.
;; WE DONT DO THIS ANYMORE
(defn move-errors-to-noir
  "Moves errors from sandbar to Noir and returns the form-data."
  [form-data errors]
  (println "(FBS) Post request failed validation! Sandbar errors (moving to noir): "
           errors "\n")
  (doseq [[field [error]] errors]
    (vali/set-error field error))
  (assoc form-data :_sandbar-errors errors))

(defn move-errors-to-flash
  "Moves the errors from sandbar validation and any form-data (values
  that were just submitted) over to the flash to be used when we
  redirect to the form page again."
  [form-data errors]
  (session/flash-put! :form-data (assoc form-data
                                   :_sandbar-errors errors)))

(defn maybe-conj
  "Gets a list containing one map or many maps. If theres only one map
  in it, return the map. Else, return all the maps in the list conj-ed
  together."
  [a]
  (if (> (count a) 1)
    (apply conj a)
    (first a)))

;;The fn below is used called by the form-helper macro when a 'form
;;function' is called, ie in a defpage, ex: (myform m "action" "/").
;;It takes in the params map m and generates a map of default
;;values. It then takes in all the validation errors in noir and
;;generates a map of errors. We cant just call (vali/on-error) for
;;each key in the map bc, for example, if the user submits a checkbox
;;or radio with NO value (empty), then it is completely left out of
;;the params map, yet might have a "cannot be nil" error in Noir. So
;;we make the default map, the errors map, and then merge them.

(defn create-errors-defaults-map
  "Used when a 'form fn' is called, either during the first time a
  form is loaded or on a render due to a validation error.  It takes
  in a map (which could be a post request map or a map of default
  values) and returns it with the form elements as keys and a values
  map with defaults and errors from Noir. Ex: {:somekey {:errors
  ['error mesage'] :default 'default message here'}"
  [default-values]
  ;;  (println "(FBS) Making form. All Noir errors: " @vali/*errors*)
  ;;  (println "(FBS) Making form. Form Params: " m)
  (let [flash-data (session/flash-get :form-data)
        flash-errors (:_sandbar-errors flash-data)
        m (if (seq flash-data)
            (dissoc flash-data :_sandbar-errors)
            default-values)
        defaults (if (seq m)
                   (maybe-conj
                    (map (fn[[k v]] {k {:errors nil :default (if (coll? v)
                                                              (map str v)
                                                              (str v))}}) m))
                   {})
        errors (if (seq flash-errors)
                 (maybe-conj
                  (map (fn[[k v]] {k {:errors (into [] v) :default ""}}) flash-errors))
                 {})
        errs-defs  (merge-with
                    (fn[a b] {:errors (:errors b) :default (:default a)})
                    defaults errors)]
       ;; (println "(FBS) Noir ERRORS: " @vali/*errors*)
       ;; (println "(FBS) Sandbar ERRORS: " (:_sandbar-errors flash-data))
    ;; (println "(FBS) Making form. Computed errors / defaults map: " errs-defs)
    errs-defs))
    
;;Takes a validator function, an url (route) to POST to, a sequence of
;;maps each containing a form element's attributes, a submit label for
;;the form, and functions to call in the POST handler on success or
;;failure
(defmacro form-helper
  "Generates a function that can be used the make the form, and registers a POST handler with Noir. "
  [sym & {:keys [fields post-url validator on-success on-failure submit-label]
          :or {on-success (constantly (response/redirect "/"))
               validator  identity}
          :as opts}] 
  (assert (and post-url fields on-success on-failure)
          "Please provide :post-url, :fields, and :on-failure to defform.")
  `(do
     (defn ~sym
       ([form-params# action# cancel-link#]
          (->> (create-errors-defaults-map form-params#)
               (assoc (-> (assoc ~opts :action action#)
                          (assoc :cancel-link cancel-link#))
                 :errors-and-defaults)
               (apply concat)
               (apply make-form))))
     (defpage [:post ~post-url] {:as m#}
       (let [errors# (~validator m#)]
         (if (empty? errors#)
           (~on-success m#)
           ((comp ~on-failure move-errors-to-flash) m# errors#))))))
