(ns de.levering-it.electric.three.utils
  (:require [hyperfiddle.electric3 :as e]
            [hyperfiddle.electric-dom3 :as dom]
            [hyperfiddle.electric-dom3-events :as events]
            [missionary.core :as m]
            [contrib.missionary-contrib :as mx]))

(def deg2rad (/ Math/PI 180))
(def deg90 (* deg2rad 90))
(def deg-90 (* deg2rad -90))
(def deg180 (* deg2rad 180))

;; keystates
#?(:cljs (defn squarewave [node enter-event leave-event init]
           (->> (mx/mix
                  (m/observe (fn [!] (events/with-listener node enter-event (fn [e] (! {(.-key e) true})))))
                  (m/observe (fn [!] (events/with-listener node leave-event (fn [e] (! {(.-key e) false}))))))
             (m/reductions {} init) (m/relieve {}))))

(e/defn PointerDown? "down flag with pointer capture on down event"
  ([]
   (PointerDown? dom/node))
  ([node]
   (e/client
     (e/input
       (->> (mx/mix
              (m/observe (fn [!] (events/with-listener node "pointerdown" (fn [e]
                                                                            (.setPointerCapture dom/node (.-pointerId e))
                                                                            (! true)))))
              (m/observe (fn [!] (events/with-listener node "pointercancel" (fn [e] (! false)))))
              (m/observe (fn [!] (events/with-listener node "pointerup" (fn [e] (! false))))))
         (m/reductions {} false)
         (m/relieve {}))))))


(e/defn RelMove  "delta movement since last move"
  ([] (RelMove dom/node))
  ([node]
   (e/client
     (e/input
       (->>  (m/observe (fn [!] (events/with-listener node "pointermove" (fn [e] (! e)))))
         (m/reductions (fn [_ e]
                         [(.-movementX e) (.-movementY e)]) [0 0])
         (m/relieve (fn [[x1 y1] [x2 y2]] [(+ x1 x2) (+ y1 y2)])))))))

(e/defn KeyDown?
  ([] (KeyDown? dom/node))
  ([node] (e/client (e/input (squarewave node "keydown" "keyup" {})))))

(e/defn KeyState []
  (e/client
    (let [a! (atom {})]
      (swap! a! merge (KeyDown?))
      (e/watch a!))))

(defn with-t->dt [f]
  (let [last! (volatile! nil)]
    (fn [e & args]
      (when-not @last!
        (vreset! last! e))
      (apply f (let [dt (- e @last!)]
                 (vreset! last! e)
                 dt) args))))

(e/defn PointerLock?
  ([] (PointerLock? dom/node))
  ([node]
   (e/client
     (e/input
       (->> (mx/mix
              (m/observe (fn [!] (events/with-listener js/document "pointerlockchange" (fn [e]
                                                                                         (if (= (.-pointerLockElement js/document) node)
                                                                                           (! true)
                                                                                           (! false))))))
              (m/observe (fn [!] (events/with-listener js/document "pointerlockerror" (fn [e]
                                                                                         (! false))))))
         (m/reductions {} (= (.-pointerLockElement js/document) node))
         (m/relieve {}))))))

#?(:cljs
   (defn fullscreen? []
     (-> (or  (.-fullscreenElement js/document)
           (.-webkitFullscreenElement js/document)
           (.-mozFullScreenElement js/document)
           (.-msFullscreenElement js/document))
       nil?
       not)))

#?(:cljs
   (defn request-fullscreen
     ([] (request-fullscreen (.-documentElement js/document)))
     ([element]
      (cond
        (.-requestFullscreen element) (.requestFullscreen element)
        (.-webkitRequestFullscreen element) (.webkitRequestFullscreen element)
        (.-mozRequestFullScreen element) (.mozRequestFullScreen element)
        (.-msRequestFullscreen element) (.msRequestFullscreen element)))))

#?(:cljs
   (defn exit-fullscreen []
     (cond
       (.-exitFullscreen js/document) (.exitFullscreen js/document)
       (.-webkitExitFullscreen js/document) (.webkitExitFullscreen js/document)
       (.-mozCancelFullScreen js/document) (.mozCancelFullScreen js/document)
       (.-msExitFullscreen js/document) (.msExitFullscreen js/document))))

(e/defn FullScreen? []
  (e/client
    (e/input
      (->> (mx/mix
             (m/observe (fn [!] (events/with-listener js/document "fullscreenchange" (fn [e]
                                                                                       (! (fullscreen?))))))
             (m/observe (fn [!] (events/with-listener js/document "webkitfullscreenchange" (fn [e]
                                                                                             (! (fullscreen?))))))
             (m/observe (fn [!] (events/with-listener js/document "mozfullscreenchange" (fn [e]
                                                                                          (! (fullscreen?))))))
             (m/observe (fn [!] (events/with-listener js/document "msfullscreenchange" (fn [e]
                                                                                         (! (fullscreen?)))))))
        (m/reductions {} (fullscreen?))
        (m/relieve {})))))

(e/defn FullscreenControl [active]
  (e/client
    (let [fullscreen? (FullScreen?)]
      (cond
        (and fullscreen? (not active)) (exit-fullscreen)
        (and (not fullscreen?) active) (request-fullscreen dom/node))
      fullscreen?)))