(ns de.levering-it.electric.three.impl
  (:require [hyperfiddle.electric3 :as e]
            [hyperfiddle.electric-dom3 :as dom]
            #?(:cljs ["three" :as three])
            [missionary.core :as m]
            [de.levering-it.electric.three.bindings :as tbp]
            [cljs.core :as c])
  #?(:cljs (:require-macros [de.levering-it.electric.three.impl])))

; internal bindings
(e/declare obj-port)
(e/declare rerender-flag)
(e/declare render-token)

(e/declare hidden-for-render)


#?(:cljs (defn interop-js
           ([^js cls]                                  #?(:cljs (new cls)))
           ([^js cls a#]                               #?(:cljs (new cls a#)))
           ([^js cls a# b#]                            #?(:cljs (new cls a# b#)))
           ([^js cls a# b# c#]                         #?(:cljs (new cls  a# b# c#)))
           ([^js cls a# b# c# d#]                      #?(:cljs (new cls a# b# c# d#)))
           ([^js cls a# b# c# d# e#]                   #?(:cljs (new cls a# b# c# d# e#)))
           ([^js cls a# b# c# d# e# f#]                #?(:cljs (new cls a# b# c# d# e# f#)))
           ([^js cls a# b# c# d# e# f# g#]             #?(:cljs (new cls a# b# c# d# e# f# g#)))
           ([^js cls a# b# c# d# e# f# g# h#]          #?(:cljs (new cls a# b# c# d# e# f# g# h#)))
           ([^js cls a# b# c# d# e# f# g# h# i#]       #?(:cljs (new cls a# b# c# d# e# f# g# h# i#)))
           ([^js cls a# b# c# d# e# f# g# h# i# j#]    #?(:cljs (new cls a# b# c# d# e# f# g# h# i# j#)))
           ([^js cls a# b# c# d# e# f# g# h# i# j# k#] #?(:cljs (new cls a# b# c# d# e# f# g# h# i# j# k#)))))

#?(:cljs (defn setlistener [^js obj val]
           (set! (.-listeners  obj) val)))

#?(:cljs (defn dispose [^js obj]
           (.dispose obj)))

(defmacro set-prop-fn [path obj]
  `(fn [val#]
     (set! (.. ~obj ~@path) val#)))

(defmacro unmount-prop [fn]
  `(new (m/observe (fn [!#] (!# nil) ~fn))))

(defn flatten-props
  ([m] (flatten-props m []))
  ([m p]
   (if (map? m)
     (mapcat
       (fn [[k v]]
         (flatten-props v (conj p k))) m)
     [[p m]])))

#?(:cljs (defn create [ctor c-args]
           (let [c-args  c-args]
             (apply interop-js ctor c-args))))

#?(:cljs (defn remove-from-parent [^js obj]
           (.removeFromParent obj)))

(e/defn SceneGraphObj [ctor c-args body]
  (let [obj (e/client (create ctor (e/$ c-args)))]
    (case (case (setlistener  obj (atom {}))
            (.add tbp/node obj))
      (do
        (e/client
          (e/on-unmount #(do
                           (remove-from-parent obj)
                           (dispose obj))))
        (binding [tbp/node obj]
          (e/$ body))))))

(e/defn ResourceObj [ctor c-args body]
  (let [obj (e/client (create ctor (e/$ c-args)))]
    (binding [tbp/node obj]
      (e/$ body))
    (e/client (e/on-unmount #(dispose obj)))
    obj))

(defn resource-obj*
  ([ctor c-args forms]
   (let [s (symbol ctor)]
     `(e/$ ResourceObj  ~s (e/fn [] [~@(take c-args forms)]) (e/fn [] (do ~@(drop c-args forms)))))))

(comment)
(defn scene-obj*
  ([ctor c-args forms]
   (let [s (symbol ctor)]
     `(e/$ SceneGraphObj  ~s (e/fn [] [~@(take c-args forms)]) (e/fn [] (do ~@(drop c-args forms)))))))

(defn size> [node]
  #?(:cljs (->> (m/observe (fn [!]
                             (! (-> node .getBoundingClientRect))
                             (let [resize-observer (js/ResizeObserver. (fn [[nd] _]
                                                                         (! (-> nd .-target .getBoundingClientRect))))]
                               (.observe resize-observer node)
                               #(.disconnect resize-observer))))
             (m/relieve {}))))

(defn -size [rect]
  #?(:cljs [(.-width rect) (.-height rect)]))

#?(:cljs (defn setPixelRatio [^js obj]
           (.setPixelRatio obj (.-devicePixelRatio js/window))))

#?(:cljs (defn render [^js renderer ^js scene ^js camera]
           (.updateProjectionMatrix camera)
           (.render renderer scene camera)))

(e/defn WebGLRenderer* [Body]
  (let [renderer (e/client (interop-js three/WebGLRenderer (clj->js {:antialias true})))
        elem (e/client (.-domElement renderer))
        w-h (e/client (-size (e/input (size> dom/node))))
        width (e/client (first w-h))
        height (e/client (second w-h))
        hidden-store (atom {})]
    (e/client (.appendChild  dom/node elem)
      (.setSize renderer width height true)
      (setPixelRatio renderer)
      (e/on-unmount #(do
                       (dispose renderer)
                       (some-> (.-parentNode elem) (.removeChild elem)))))
    (binding [tbp/renderer renderer
              tbp/view-port-width width
              tbp/view-port-height height
              dom/node elem
              hidden-for-render hidden-store]
      (Body))))

#?(:cljs (defn update-camera [^js camera]
           (.updateProjectionMatrix camera)))

#?(:cljs (defn -render-for-cam [renderer scene camera hidden-for-render]
           (if (instance? three/CubeCamera camera)
             (do
               (doseq [f (keys hidden-for-render)]
                 (f renderer scene camera))
               (.update camera renderer scene)
               (doseq [f (vals hidden-for-render)]
                 (f)))
             (do
               (update-camera camera)
               (doseq [f (keys hidden-for-render)]
                 (f renderer scene camera))
               (render renderer scene camera)
               (doseq [f (vals hidden-for-render)]
                 (f))))))

#?(:cljs (defn render-scene [renderer scene camera token hidden-for-render]
           (when token
             (if (vector? camera)
               (doseq [c  camera]
                 (-render-for-cam renderer scene c @hidden-for-render))
               (-render-for-cam renderer scene camera @hidden-for-render)))))

(e/defn Scene* [Body]
  (let [scene (e/client (interop-js three/Scene))]
    (e/on-unmount #(dispose scene))
    (case (setlistener  scene (atom {}))
      (binding [tbp/node scene
                tbp/scene scene]
        (let [r (Body)]
          (e/client
            (render-scene tbp/renderer scene tbp/camera (e/System-time-ms) hidden-for-render))
          r)))))

(defn props* [m]
  `(do
     ~@(map (fn [[k v]]
              (let [path (map #(symbol (str "-" (name %))) k)]
                `(let [org-val# (.. tbp/node ~@path)]
                   ((set-prop-fn ~path tbp/node) ~v)
                   (e/on-unmount #(do
                                    ((set-prop-fn ~path tbp/node) org-val#))))))
         (sort-by first (flatten-props m)))))

(e/defn HideForCamera* [camera Body]
  (letfn [(hide [renderer scene camera']
            (when (and (= renderer tbp/renderer) (= scene tbp/scene) (= camera camera'))
              (set! (.-visible tbp/node) false)))]
    (swap! hidden-for-render assoc hide  #(set! (.-visible tbp/node) true))
    (e/on-unmount #(swap! hidden-for-render dissoc hide))
    (Body)))

(e/defn AddObj [obj Body]
  (e/client
    (case (case (setlistener  obj (atom {}))
            (.add tbp/node obj))
      (do
        (e/client
          (e/on-unmount #(do
                           (remove-from-parent obj)
                           (dispose obj))))
        (binding [tbp/node obj]
          (e/$ Body))))))