(ns de.phenomdevel.components.discord-bot-receiver
  (:require
   [clojure.pprint :as pprint]
   [clojure.core.async :as a]

   [medley.core :as mc]
   [taoensso.timbre :as log]
   [discljord.events :as dise]
   [discljord.events.state :as diss]
   [com.stuartsierra.component :as c]))


;; =============================================================================
;; Private Helper

(defn- pp-str
  [x]
  (-> x
      (pprint/pprint)
      (with-out-str)))

(def ^:private termination-keys
  #{:disconnect})

(defn- terminate?
  [[event-type _event-data]]
  (termination-keys event-type))

(defn- user-event?
  [[_event-type event-data]]
  (let [bot-member?
        (get-in event-data [:member :user :bot])

        bot-user?
        (get-in event-data [:author :bot])

        bot?
        (or bot-user? bot-member?)]

    (not bot?)))

(defn- listen
  [dependencies guild-state receiver-ch event-handler]
  (a/go-loop [event-count
              1]

    (when-let [[event-type event-data :as event]
               (a/<! receiver-ch)]

      (try
        (if-not (user-event? event)
          (log/trace (format "[DiscordBotReceiver] <%s> Event `%s` is skipped because it is from a bot user." event-count event-type))
          (do
            (log/info (format "[DiscordBotReceiver] <%s> New event `%s`" event-count event-type))
            (log/trace (format "[DiscordBotReceiver] <%s>\n%s" event-count (pp-str event)))

            (dise/dispatch-handlers #'diss/caching-handlers event-type event-data guild-state)
            (event-handler dependencies event)))

        ;; TODO Core-Async Exception Handling...
        (catch Throwable ex
          (log/error (format "[DiscordBotReceiver] <%s> Exception while handling event `%s`\n%s" event-count event-type (pp-str ex)))))

      (when-not (terminate? event)
        (recur
         (inc event-count))))))


;; =============================================================================
;; Component

;; TODO: Add error messages if dependencies are not supplied
(defrecord DiscordBotReceiver [token event-handler listen-ch discord-bot-sender discord-bot-connection discord-bot-guild-state]
  c/Lifecycle
  (start
   [this]
   (when-let [receiver-ch
              (:receiver-ch discord-bot-connection)]

     (let [guild-state
           (:guild-state discord-bot-guild-state)

           dependencies
           (mc/filter-vals record? this)

           listen-ch
           (listen dependencies guild-state receiver-ch event-handler)]

       (log/info "[DiscordBotReceiver] Started.")
       (assoc this :listen-ch listen-ch))))

  (stop
   [this]
   (when listen-ch
     (a/close! listen-ch))
   (log/info "[DiscordBotReceiver] Stopped.")
   (assoc this :listen-ch nil)))


;; =============================================================================
;; Public API

(defn new-discord-bot-receiver
  "Takes a `config` and an `event-handler` to create a new discord-bot-receiver.
  `config` may contain keys
    `token` - Discord guild token to connect to
  `event-handler` should be a function which will be supplied with
    `context` - Will hold all supplied dependencies on `discord-bot-receiver`
  and the upcoming event in form of
    `[event-type event-data]`"
  [config event-handler]
  (-> config
      (assoc :event-handler event-handler)
      (map->DiscordBotReceiver)))
